/*
 * Copyright 2013 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.client.basic;

import java.net.URISyntaxException;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Session;
import org.jflux.api.core.util.DefaultTimestampSource;
import org.jflux.api.core.util.EmptyAdapter;
import org.robokind.api.messaging.MessageAsyncReceiver;
import org.robokind.api.messaging.MessageSender;
import org.robokind.api.sensor.DeviceBoolEvent;
import org.robokind.api.sensor.gpio.GpioService;
import org.robokind.api.sensor.gpio.RemoteGpioService;
import org.robokind.avrogen.sensor.DeviceBoolRecord;
import org.robokind.avrogen.sensor.HeaderRecord;
import org.robokind.client.basic.ConnectionContext.RkServiceConnector;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
final class RkSensorConnector extends RkServiceConnector{
    final static String GPIO_VALUE_RECEIVER = "gpioValueReceiver";
    final static String GPIO_DIRECTION_SENDER = "gpioDirectionSender";
    final static String GPIO_VALUE_SENDER = "gpioValueSender";
    
    private static RkSensorConnector theRkSensorConnector;
    
    private String theGpioInputDest = "GpioEvent";
    private String theGpioDirectionDest = "GpioCommand";
    private String theGpioOutputDest = "GpioWriteCommand";
    
    static synchronized RkSensorConnector getConnector(){
        if(theRkSensorConnector == null){
            theRkSensorConnector = new RkSensorConnector();
        }
        return theRkSensorConnector;
    }
    
    @Override
    protected synchronized void addConnection(Session session) 
            throws JMSException, URISyntaxException{
        if(myConnectionContext == null || myConnectionsFlag){
            return;
        }
        Destination gpioValReceiver = ConnectionContext.getQueue(theGpioInputDest);
        myConnectionContext.addAsyncReceiver(GPIO_VALUE_RECEIVER, session, gpioValReceiver,
                DeviceBoolRecord.class, DeviceBoolRecord.SCHEMA$,
                new EmptyAdapter<DeviceBoolRecord, DeviceBoolRecord>());
        Destination gpioDirSender = ConnectionContext.getQueue(theGpioDirectionDest);
        myConnectionContext.addSender(GPIO_DIRECTION_SENDER, session, gpioDirSender, 
                new EmptyAdapter<DeviceBoolRecord, DeviceBoolRecord>());
        Destination gpioValSender = ConnectionContext.getQueue(theGpioOutputDest);
        myConnectionContext.addSender(GPIO_VALUE_SENDER, session, gpioValSender, 
                new EmptyAdapter<DeviceBoolRecord, DeviceBoolRecord>());
        myConnectionsFlag = true;
    }
    
    synchronized RemoteGpioService buildRemoteClient() {
        if(myConnectionContext == null || !myConnectionsFlag){
            return null;
        }
        MessageAsyncReceiver<DeviceBoolEvent<HeaderRecord>> gpioValReceiver = 
                myConnectionContext.getAsyncReceiver(GPIO_VALUE_RECEIVER);
        MessageSender<DeviceBoolEvent<HeaderRecord>> gpioDirSender = 
                myConnectionContext.getSender(GPIO_DIRECTION_SENDER);
        MessageSender<DeviceBoolEvent<HeaderRecord>> gpioValSender = 
                myConnectionContext.getSender(GPIO_VALUE_SENDER);
        
        HeaderRecord.Builder headerFact = HeaderRecord.newBuilder();
        headerFact.setFrameId(0);
        headerFact.setSequenceId(0);
        headerFact.setTimestamp(0);
        HeaderRecord emptyHeader = headerFact.build();
        
        DeviceBoolRecord.Builder eventFact = DeviceBoolRecord.newBuilder();
        eventFact.setHeader(emptyHeader);
        eventFact.setChannelId(0);
        eventFact.setBoolValue(false);
        
        RemoteGpioService<HeaderRecord> client = new RemoteGpioService<HeaderRecord>(
                headerFact, new DefaultTimestampSource(), eventFact,  
                gpioDirSender, gpioValSender, gpioValReceiver, 0, 0, 0, 
                0, 1, 2, 3, 4, 5, 8, 9, 10, 11);
        return client;
    }
    
    static void initializeGpioClient(GpioService client){
        client.setPinDirection(8, GpioService.OUT);
        client.setPinValue(8, true);
        client.setPinDirection(10, GpioService.OUT);
        client.setPinValue(10, true);
        for(int i : RobotSensors.ALL_GPIO_PINS){
            client.setPinDirection(i, GpioService.IN);
        }
    }    
}
