/*
 * Copyright 2013 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.client.basic;

import java.net.URISyntaxException;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Session;
import org.jflux.api.core.util.EmptyAdapter;
import org.robokind.api.messaging.MessageAsyncReceiver;
import org.robokind.api.messaging.MessageSender;
import org.robokind.api.sensor.DeviceReadPeriodEvent;
import org.robokind.api.sensor.FilteredVector3Event;
import org.robokind.api.sensor.GyroConfigEvent;
import org.robokind.api.sensor.imu.RemoteGyroscopeServiceClient;
import org.robokind.avrogen.sensor.DeviceReadPeriodRecord;
import org.robokind.avrogen.sensor.FilteredVector3Record;
import org.robokind.avrogen.sensor.GyroConfigRecord;
import org.robokind.avrogen.sensor.HeaderRecord;
import org.robokind.client.basic.ConnectionContext.RkServiceConnector;

/**
 *
 * @author Jason G. Pallack <jgpallack@gmail.com>
 */
final class RkGyroscopeConnector extends RkServiceConnector{
    final static String GYRO_VALUE_RECEIVER = "gyroValueReceiver";
    final static String GYRO_CONFIG_SENDER = "gyroConfigSender";
    final static String GYRO_READ_PERIOD_SENDER = "gyroReadPeriodSender";
    
    private static RkGyroscopeConnector theRkGyroscopeConnector;
    
    private String theGyroInputDest = "gyroEvent";
    private String theGyroConfigDest = "gyroConfig";
    private String theGyroReadDest = "gyroRead";
    
    static synchronized RkGyroscopeConnector getConnector(){
        if(theRkGyroscopeConnector == null){
            theRkGyroscopeConnector = new RkGyroscopeConnector();
        }
        return theRkGyroscopeConnector;
    }
    
    @Override
    protected synchronized void addConnection(Session session) 
            throws JMSException, URISyntaxException{
        if(myConnectionContext == null || myConnectionsFlag){
            return;
        }
        Destination gyroValReceiver = ConnectionContext.getTopic(theGyroInputDest);
        myConnectionContext.addAsyncReceiver(GYRO_VALUE_RECEIVER, session, gyroValReceiver,
                FilteredVector3Record.class, FilteredVector3Record.SCHEMA$,
                new EmptyAdapter<FilteredVector3Record, FilteredVector3Record>());
        Destination gyroCfgSender = ConnectionContext.getTopic(theGyroConfigDest);
        myConnectionContext.addSender(GYRO_CONFIG_SENDER, session, gyroCfgSender, 
                new EmptyAdapter<GyroConfigRecord, GyroConfigRecord>());
        Destination gyroPerSender = ConnectionContext.getTopic(theGyroReadDest);
        myConnectionContext.addSender(GYRO_READ_PERIOD_SENDER, session, gyroPerSender, 
                new EmptyAdapter<DeviceReadPeriodRecord, DeviceReadPeriodRecord>());
        myConnectionsFlag = true;
    }
    
    synchronized RemoteGyroscopeServiceClient buildRemoteClient() {
        if(myConnectionContext == null || !myConnectionsFlag){
            return null;
        }
        MessageAsyncReceiver<FilteredVector3Event> gyroValReceiver = 
                myConnectionContext.getAsyncReceiver(GYRO_VALUE_RECEIVER);
        MessageSender<GyroConfigEvent<HeaderRecord>> gyroCfgSender = 
                myConnectionContext.getSender(GYRO_CONFIG_SENDER);
        MessageSender<DeviceReadPeriodEvent<HeaderRecord>> gyroPerSender = 
                myConnectionContext.getSender(GYRO_READ_PERIOD_SENDER);
        
        RemoteGyroscopeServiceClient<HeaderRecord> client =
                new RemoteGyroscopeServiceClient<HeaderRecord>(
                gyroCfgSender, gyroPerSender, gyroValReceiver);
        return client;
    }
}
