/*
 * Copyright 2013 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.client.basic;

import java.net.URISyntaxException;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Session;
import org.jflux.api.core.util.DefaultTimestampSource;
import org.jflux.api.core.util.EmptyAdapter;
import org.robokind.api.messaging.MessageAsyncReceiver;
import org.robokind.api.messaging.MessageSender;
import org.robokind.api.sensor.DeviceBoolEvent;
import org.robokind.api.sensor.DeviceReadPeriodEvent;
import org.robokind.api.sensor.GpioConfigEvent;
import org.robokind.api.sensor.gpio.GpioService;
import org.robokind.api.sensor.gpio.RemoteGpioServiceClient;
import org.robokind.client.basic.ConnectionContext.RkServiceConnector;
import org.robokind.impl.sensor.DeviceBoolRecord;
import org.robokind.impl.sensor.DeviceReadPeriodRecord;
import org.robokind.impl.sensor.GpioConfigRecord;
import org.robokind.impl.sensor.HeaderRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
final class RkSensorConnector extends RkServiceConnector{
    final static String GPIO_VALUE_RECEIVER = "gpioValueReceiver";
    final static String GPIO_DIRECTION_SENDER = "gpioDirectionSender";
    final static String GPIO_READ_PERIOD_SENDER = "gpioReadPeriodSender";
    final static String GPIO_VALUE_SENDER = "gpioValueSender";
    
    private static RkSensorConnector theRkSensorConnector;
    
    private String theGpioInputDest = "gpioEvent";
    private String theGpioDirectionDest = "gpioConfig";
    private String theGpioOutputDest = "gpioWrite";
    private String theGpioReadDest = "gpioRead";
    
    static synchronized RkSensorConnector getConnector(){
        if(theRkSensorConnector == null){
            theRkSensorConnector = new RkSensorConnector();
        }
        return theRkSensorConnector;
    }
    
    @Override
    protected synchronized void addConnection(Session session) 
            throws JMSException, URISyntaxException{
        if(myConnectionContext == null || myConnectionsFlag){
            return;
        }
        Destination gpioValReceiver = ConnectionContext.getTopic(theGpioInputDest);
        myConnectionContext.addAsyncReceiver(GPIO_VALUE_RECEIVER, session, gpioValReceiver,
                DeviceBoolRecord.class, DeviceBoolRecord.SCHEMA$,
                new EmptyAdapter<DeviceBoolRecord, DeviceBoolRecord>());
        Destination gpioDirSender = ConnectionContext.getTopic(theGpioDirectionDest);
        myConnectionContext.addSender(GPIO_DIRECTION_SENDER, session, gpioDirSender, 
                new EmptyAdapter<GpioConfigRecord, GpioConfigRecord>());
        Destination gpioPerSender = ConnectionContext.getTopic(theGpioReadDest);
        myConnectionContext.addSender(GPIO_READ_PERIOD_SENDER, session, gpioPerSender, 
                new EmptyAdapter<DeviceReadPeriodRecord, DeviceReadPeriodRecord>());
        Destination gpioValSender = ConnectionContext.getTopic(theGpioOutputDest);
        myConnectionContext.addSender(GPIO_VALUE_SENDER, session, gpioValSender, 
                new EmptyAdapter<DeviceBoolRecord, DeviceBoolRecord>());
        myConnectionsFlag = true;
    }
    
    synchronized RemoteGpioServiceClient buildRemoteClient() {
        if(myConnectionContext == null || !myConnectionsFlag){
            return null;
        }
        MessageAsyncReceiver<DeviceBoolEvent<HeaderRecord>> gpioValReceiver = 
                myConnectionContext.getAsyncReceiver(GPIO_VALUE_RECEIVER);
        MessageSender<GpioConfigEvent<HeaderRecord>> gpioDirSender = 
                myConnectionContext.getSender(GPIO_DIRECTION_SENDER);
        MessageSender<DeviceReadPeriodEvent<HeaderRecord>> gpioPerSender = 
                myConnectionContext.getSender(GPIO_READ_PERIOD_SENDER);
        MessageSender<DeviceBoolEvent<HeaderRecord>> gpioValSender = 
                myConnectionContext.getSender(GPIO_VALUE_SENDER);
        
        HeaderRecord.Builder headerFact = HeaderRecord.newBuilder();
        headerFact.setFrameId(0);
        headerFact.setSequenceId(0);
        headerFact.setTimestamp(0);
        HeaderRecord emptyHeader = headerFact.build();
        
        DeviceBoolRecord.Builder eventFact = DeviceBoolRecord.newBuilder();
        eventFact.setHeader(emptyHeader);
        eventFact.setChannelId(0);
        eventFact.setBoolValue(false);
        
        GpioConfigRecord.Builder configFact = GpioConfigRecord.newBuilder();
        configFact.setHeader(emptyHeader);
        configFact.setInputMask(0);
        
        RemoteGpioServiceClient<HeaderRecord> client = new RemoteGpioServiceClient<HeaderRecord>(
                headerFact, new DefaultTimestampSource(), eventFact, configFact,  
                gpioDirSender, gpioPerSender, gpioValSender, gpioValReceiver, 0,
                0, 0, 0, 1, 2, 3, 4, 5, 8, 9, 10, 11);
        return client;
    }
    
    static void initializeGpioClient(GpioService client){
        client.setPinDirection(8, GpioService.OUT);
        client.setPinValue(8, true);
        client.setPinDirection(10, GpioService.OUT);
        client.setPinValue(10, true);
        for(int i : RobotSensors.ALL_GPIO_PINS){
            client.setPinDirection(i, GpioService.IN);
        }
    }    
}
