/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.animation.messaging;

import java.util.logging.Logger;
import javax.jms.Destination;
import javax.jms.Session;
import org.robokind.api.animation.Animation;
import org.robokind.api.animation.protocol.AnimationEvent;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.avrogen.animation.AnimationEventRecord;
import org.robokind.impl.messaging.JMSAvroMessageSender;

/**
 *
 * JMSAvroMessageSender for sending AnimationEvents.
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSAnimationSender 
        extends JMSAvroMessageSender<
                AnimationEvent, 
                AnimationEventRecord, 
                Listener<AnimationEvent>> {
    static final Logger theLogger = Logger.getLogger(JMSAnimationSender.class.getName());
    private String mySourceId;
    private String myDestinationId;
    
    /**
     * Creates an empty JMSAnimationSender
     * @param session JMS Session to use
     * @param destination JMS Destination to use
     */
    public JMSAnimationSender(Session session, Destination destination){
        super(session, destination);
        setAdapter(new AnimationEventAdapter());
    }
    /**
     * Sets the source id String for this sender.
     * This is used as the sourceId in outgoing AnimationEvents.
     * @param sourceId source id String to use
     */
    public void setSourceId(String sourceId){
        mySourceId = sourceId;
    }
    /**
     * Set the destination id String for this sender.
     * This is used as the destinationId in outgoing AnimationEvents.
     * @param destId destination id String to use.
     */
    public void setDestinationId(String destId){
        myDestinationId = destId;
    }
    /**
     * Creates an AnimationEvent for the given Animation, and sends it.
     * @param animation Animation to send
     * @throws NullPointerException if the sourceId or destinationId for this
     * sender are null, or if the given Animation is null.
     */
    public void sendAnimation(Animation animation){
        if(animation == null || mySourceId == null || myDestinationId == null){
            throw new NullPointerException();
        }
        AnimationEvent frameEvent = 
                new PortableAnimationEvent(
                        mySourceId, myDestinationId, TimeUtils.now(), animation);
        super.sendMessage(frameEvent);
    }
    /**
     * Converts AnimationEvents to AnimationEventRecords
     */
    public static class AnimationEventAdapter 
            implements Adapter<AnimationEvent, AnimationEventRecord>{
        @Override
        public AnimationEventRecord adapt(AnimationEvent a) {
            return new PortableAnimationEvent(a).getAnimationEventRecord();
        }
    }
}
