/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.animation.osgi;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.jms.Connection;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Session;
import org.robokind.api.animation.messaging.RemoteAnimationPlayerHost;
import org.robokind.api.animation.player.AnimationPlayer;
import org.robokind.api.common.osgi.AbstractLifecycleProvider;
import org.robokind.api.common.osgi.DescriptorListBuilder;
import org.robokind.impl.animation.messaging.JMSRemoteAnimationPlayerHost;
import org.robokind.impl.messaging.utils.ConnectionManager;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSAnimationPlayerLifecycleProvider extends 
        AbstractLifecycleProvider<RemoteAnimationPlayerHost, JMSRemoteAnimationPlayerHost> {
    private final static Logger theLogger = 
            Logger.getLogger(JMSAnimationPlayerLifecycleProvider.class.getName());
    private final static String theDepPlayerId = "animPlayer";
    private final static String theDepConnectionId = "animConnection";
    private final static String theDepAnimationDestId = "animDestination";
    
    private Session mySession;
    
    public JMSAnimationPlayerLifecycleProvider(String animPlayerId, 
            String connId, String animDestId){
        super(new DescriptorListBuilder()
                .addId(theDepPlayerId, AnimationPlayer.class, 
                        AnimationPlayer.PROP_PLAYER_ID, animPlayerId)
                .addId(theDepConnectionId, Connection.class, 
                        ConnectionManager.PROP_CONNECTION_ID, connId)
                .addId(theDepAnimationDestId, Destination.class, 
                        ConnectionManager.PROP_DESTINATION_ID, animDestId)
                .getDescriptors());
    }

    @Override
    protected JMSRemoteAnimationPlayerHost create(Map<String, Object> services){
        AnimationPlayer player = (AnimationPlayer)services.get(theDepPlayerId);
        Connection con = (Connection)services.get(theDepConnectionId);
        Destination dest = (Destination)services.get(theDepAnimationDestId);
        
        setSession(con);
        try{
            return new JMSRemoteAnimationPlayerHost(
                    mySession, dest, player);
        }catch(Exception ex){
            theLogger.log(Level.WARNING, 
                    "There was an error creating and "
                    + "starting the RemoteAnimationPlayerHost.", ex);
            return null;
        }
        
    }
    
    private void setSession(Connection con){
        Session oldSession = mySession;
        try{
            mySession = con.createSession(false, Session.CLIENT_ACKNOWLEDGE);
        }catch(JMSException ex){
            theLogger.log(Level.WARNING, "Unable to create Session.", ex);
            mySession = null;
        }
        if(myService != null){
            try{
                myService.setSession(mySession);
            }catch(Exception ex){
                theLogger.log(Level.WARNING, 
                        "There was an error setting the Session.", ex);
            }
        }
        if(oldSession != null){
            try{
                oldSession.close();
            }catch(Exception ex){
                theLogger.log(Level.WARNING, 
                        "There was an error closing the old Session.", ex);
            }
        }
    }
    
    @Override
    protected void handleChange(String serviceId, Object service) {
        if(theDepPlayerId.equals(serviceId)){
            myService.setAnimationPlayer((AnimationPlayer)service);
        }else if(theDepConnectionId.equals(serviceId)){
            setSession((Connection)service);
        }else if(theDepAnimationDestId.equals(serviceId)){
            try{
                myService.setDestination((Destination)service);
            }catch(Exception ex){
                theLogger.log(Level.WARNING, 
                        "There was an error setting the new Destination.", ex);
            }
        }
    }
    
    @Override
    public Class<RemoteAnimationPlayerHost> getServiceClass() {
        return RemoteAnimationPlayerHost.class;
    }
}
