/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.impl.animation.xml;

import org.robokind.api.common.playable.Playable;
import org.robokind.api.common.services.ServiceConfigurationLoader;
import org.robokind.api.common.services.ServiceFactory;
import org.robokind.api.common.services.addon.AddOnUtils;
import org.robokind.api.common.services.addon.ServiceAddOnDriver;
import java.util.logging.Level;
import org.robokind.api.common.services.addon.ServiceAddOn;
import java.awt.geom.Point2D;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.util.logging.Logger;
import org.robokind.api.animation.xml.AnimationFileWriter;
import org.robokind.api.animation.Animation;
import org.robokind.api.animation.Channel;
import org.robokind.api.animation.MotionPath;
import org.robokind.extern.utils.xpp3.XMLUtils;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlSerializer;
import static org.robokind.api.animation.xml.AnimationXML.*;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class XPP3AnimationXMLWriter implements AnimationFileWriter{
    private final static Logger theLogger = Logger.getLogger(XPP3AnimationXMLWriter.class.getName());
    private static String namespace = null;

    @Override
    public void writeAnimation(String path, Animation anim) throws Exception{
        XPP3AnimationXMLWriter.saveAnimation(path, anim);
    }
    
    /**
     * Saves an Animation to disk as an XML file.
     * @param file the full path to the destination file
     * @param a the Animation to save
     * @throws XmlPullParserException
     * @throws IOException
     * @throws IllegalArgumentException
     * @throws FileNotFoundException
     */
    static void saveAnimation(String file, Animation a) throws
            XmlPullParserException, IOException, IllegalArgumentException, FileNotFoundException{
        XmlSerializer xs = XMLUtils.getXmlFileSerializer(file);
        xs.startDocument(null, null);
        xs.text("\n");      //new line after xml version tag.
        writeAnimation(xs, a, file);
        xs.endDocument();
        xs.flush();
    }

    /**
     *
     * @param xs
     * @param a
     * @throws IOException
     */
    public static void writeAnimation(XmlSerializer xs, Animation a, String file) throws IOException{
        xs.startTag(namespace, ANIMATION);
            XMLUtils.writeVersionProperty(xs, a.getVersion(), ANIMATION_VERSION_TYPE);
            xs.startTag(namespace, CHANNELS);
                for(Channel channel : a.getChannels()){
                    writeChannel(xs, channel);
                }
            xs.endTag(namespace, CHANNELS);
            writeAddOnList(xs, a.getAddOns(), file);
        xs.endTag(namespace, ANIMATION);
    }

    /**
     *
     * @param xs
     * @param channel
     * @throws IOException
     */
    public static void writeChannel(XmlSerializer xs, Channel channel) throws IOException{
        xs.startTag(namespace, CHANNEL);
            xs.attribute(namespace, CHANNEL_ID, channel.getId().toString());
            String name = channel.getName();
            if(name != null && !name.isEmpty()){
                xs.attribute(namespace, CHANNEL_NAME, name);
            }
            xs.startTag(namespace, MOTION_PATHS);
                for(MotionPath mp : channel.getMotionPaths()){
                    writeMotionPath(xs, mp);
                }
            xs.endTag(namespace, MOTION_PATHS);
        xs.endTag(namespace, CHANNEL);
    }

    /**
     *
     * @param xs
     * @param mp
     * @throws IOException
     */
    public static void writeMotionPath(XmlSerializer xs, MotionPath mp) throws IOException{
        xs.startTag(namespace, MOTION_PATH);
            String name = mp.getName();
            if(name != null && !name.isEmpty()){
                xs.attribute(namespace, MOTION_PATH_NAME, name);
            }
            XMLUtils.writeVersionProperty(xs, mp.getInterpolatorVersion(), INTERPOLATION_VERSION_TYPE);
            writeControlPoints(xs, mp.getControlPoints());
        xs.endTag(namespace, MOTION_PATH);
    }

    /**
     *
     * @param xs
     * @param points
     * @throws IOException
     */
    public static void writeControlPoints(XmlSerializer xs, List<Point2D> points) throws IOException{
        xs.startTag(namespace, CONTROL_POINTS);
        for(Point2D p : points){
            writeControlPoint(xs, p);
        }
        xs.endTag(namespace, CONTROL_POINTS);
    }

    /**
     *
     * @param xs
     * @param p
     * @throws IOException
     */
    public static void writeControlPoint(XmlSerializer xs, Point2D p) throws IOException{
        xs.startTag(namespace, CONTROL_POINT);
            XMLUtils.format(xs, false);
            xs.startTag(namespace, TIME);
                xs.text(((Double)p.getX()).toString());
            xs.endTag(namespace, TIME);
            xs.startTag(namespace, POSITION);
                xs.text(((Double)p.getY()).toString());
            xs.endTag(namespace, POSITION);
            XMLUtils.format(xs, true);
        xs.endTag(namespace, CONTROL_POINT);
    }
    
    public static void writeAddOnList(XmlSerializer xs,
            List<ServiceAddOn<Playable>> addons, String animPath) throws IOException{
        if(addons == null || animPath == null){
            throw new NullPointerException();
        }
        xs.startTag(namespace, ADDONS);
        int addonCount = 0;
        for(ServiceAddOn addon : addons){
            String addonPath = animPath + ".addon." + addonCount + ".conf";
            try{
                writeAddOn(xs, addon, addonPath);
            }catch(Exception ex){
                theLogger.log(Level.WARNING, "Error writing AddOn.", ex);
                continue;
            }
            addonCount++;
        }
        xs.endTag(namespace, ADDONS);
    }
    
    public static void writeAddOn(XmlSerializer xs, 
            ServiceAddOn<Playable> addon, String addonPath) throws Exception{
        if(addon == null || addonPath == null){
            return;
        }
        ServiceAddOnDriver driver = addon.getAddOnDriver();
        if(driver == null){
            throw new NullPointerException();
        }
        if(!AddOnUtils.saveAddOnConfig(addon, addonPath)){
            return;
        }
        xs.startTag(namespace, ADDON);
            XMLUtils.writeVersionProperty(xs, 
                    driver.getServiceVersion(), 
                    ServiceFactory.PROP_SERVICE_VERSION);
            XMLUtils.writeVersionProperty(xs, 
                    driver.getConfigurationFormat(), 
                    ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION);
            xs.startTag(namespace, ADDON_FILE);
                xs.text(addonPath);
            xs.endTag(namespace, ADDON_FILE);
        xs.endTag(namespace, ADDON);
    }
}
