/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.audio.config;

import java.io.File;
import org.apache.avro.util.Utf8;
import org.robokind.api.audio.config.WavPlayerConfig;
import org.robokind.api.common.config.VersionProperty;
import org.robokind.avrogen.audio.WavPlayerConfigRecord;
import org.robokind.bind.apache_avro.AvroUtils;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableWavPlayerConfig implements WavPlayerConfig{
    
    private WavPlayerConfigRecord myConfigRecord;
    
    public PortableWavPlayerConfig(WavPlayerConfigRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myConfigRecord = record;
    }
    
    public PortableWavPlayerConfig(WavPlayerConfig config){
        if(config == null){
            throw new NullPointerException();
        }
        if(config instanceof PortableWavPlayerConfig){
            myConfigRecord = ((PortableWavPlayerConfig)config).myConfigRecord;
            return;
        }
        myConfigRecord = new WavPlayerConfigRecord();
        myConfigRecord.wavPlayerId = new Utf8(config.getWavPlayerId());
        myConfigRecord.wavLocation = new Utf8(config.getWavLocation());
        myConfigRecord.startTimeMicrosec = config.getStartTimeMicrosec();
        myConfigRecord.stopTimeMicrosec = config.getStopTimeMicrosec();
        myConfigRecord.startDelayMillisec = config.getStartDelayMillisec();
    }
    
    public PortableWavPlayerConfig(String playerId, String wavLocation,
            long startTimeMicrosec, long stopTimeMicrosec, 
            long startDelayMillisec){
        if(playerId == null || wavLocation == null){
            throw new NullPointerException();
        }
        myConfigRecord = new WavPlayerConfigRecord();
        myConfigRecord.wavPlayerId = new Utf8(playerId);
        myConfigRecord.wavLocation = new Utf8(wavLocation);
        myConfigRecord.startTimeMicrosec = startTimeMicrosec;
        myConfigRecord.stopTimeMicrosec = stopTimeMicrosec;
        myConfigRecord.startDelayMillisec = startDelayMillisec;
    }
    
    @Override
    public String getWavPlayerId() {
        return myConfigRecord.wavPlayerId.toString();
    }

    @Override
    public String getWavLocation() {
        return myConfigRecord.wavLocation.toString();
    }

    @Override
    public long getStartTimeMicrosec() {
        return myConfigRecord.startTimeMicrosec;
    }

    @Override
    public long getStopTimeMicrosec() {
        return myConfigRecord.stopTimeMicrosec;
    }

    @Override
    public long getStartDelayMillisec() {
        return myConfigRecord.startDelayMillisec;
    }
    
    public WavPlayerConfigRecord getRecord(){
        return myConfigRecord;
    }
    
    public static void writeToFile(File file, WavPlayerConfig config) throws Exception {
        if(file == null || config == null){
            throw new NullPointerException();
        }
        PortableWavPlayerConfig portableConfig;
        if(config instanceof PortableWavPlayerConfig){
            portableConfig = (PortableWavPlayerConfig)config;
        }else{
            portableConfig = new PortableWavPlayerConfig(config);
        }
        WavPlayerConfigRecord record = portableConfig.getRecord();
        AvroUtils.writeToFile(
                record, WavPlayerConfigRecord.SCHEMA$, file, true);
    
    }
}
