/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import javax.jms.BytesMessage;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.Session;
import org.apache.qpid.client.message.JMSBytesMessage;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.Listener;
import org.robokind.api.messaging.DefaultMessageBlockingReceiver;
import org.robokind.api.motion.messaging.messages.RobotResponse;
import org.robokind.avrogen.motion.RobotDefinitionResponseRecord;
import org.robokind.avrogen.motion.RobotPositionResponseRecord;
import org.robokind.avrogen.motion.RobotStatusResponseRecord;
import org.robokind.bind.apache_qpid.QpidUtils;
import org.robokind.impl.messaging.JMSBytesRecordBlockingReceiver;
import org.robokind.impl.motion.messaging.command.PortableRobotDefinitionResponse;
import org.robokind.impl.motion.messaging.command.PortableRobotPositionResponse;
import org.robokind.impl.motion.messaging.command.PortableRobotStatusResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSRobotResponseBlockingReceiver 
        extends DefaultMessageBlockingReceiver<
                RobotResponse, BytesMessage, Listener<RobotResponse>> {
    private final static Logger theLogger = 
            LoggerFactory.getLogger(JMSRobotResponseBlockingReceiver.class);
    public final static String MIME_ROBOT_DEFINITION_RESPONSE = "application/avro-robot-def-resp";
    public final static String MIME_ROBOT_STATUS_RESPONSE = "application/avro-robot-stat-resp";
    public final static String MIME_ROBOT_POSITION_RESPONSE = "application/avro-robot-pos-resp";
    
    private Session mySession;
    private Destination myDestination;
    private JMSBytesRecordBlockingReceiver myReceiver;
    
    public JMSRobotResponseBlockingReceiver(
            Session session, Destination destination) throws JMSException{
        mySession = session;
        myDestination = destination;
        MessageConsumer consumer = mySession.createConsumer(myDestination);
        myReceiver = new JMSBytesRecordBlockingReceiver(consumer);
        setRecordReceiver(myReceiver);
        setAdapter(new RobotResponseRecordAdapter());
    }
    
    class RobotResponseRecordAdapter 
            implements Adapter<BytesMessage, RobotResponse>{

        @Override
        public RobotResponse adapt(BytesMessage a) {
            String mime = ((JMSBytesMessage)a).getEncoding();
            if(mime == null || mime.isEmpty()){
                return null;
            }else if(MIME_ROBOT_DEFINITION_RESPONSE.equals(mime)){
                return getDefinitionResponse(a);
            }else if(MIME_ROBOT_STATUS_RESPONSE.equals(mime)){
                return getStatusResponse(a);
            }else if(MIME_ROBOT_POSITION_RESPONSE.equals(mime)){
                return getPositionResponse(a);
            }
            return null;
        }
        
        private RobotResponse getDefinitionResponse(BytesMessage bmsg){
            try{
                RobotDefinitionResponseRecord record = 
                        QpidUtils.unpackAvroMessage(
                                RobotDefinitionResponseRecord.class, 
                                null, 
                                RobotDefinitionResponseRecord.SCHEMA$, 
                                bmsg);
                return new PortableRobotDefinitionResponse(record);
            }catch(Exception ex){
                theLogger.error("Error deserializing Avro Record.", ex);
                return null;
            }
        }
        
        private RobotResponse getStatusResponse(BytesMessage bmsg){
            try{
                RobotStatusResponseRecord record = 
                        QpidUtils.unpackAvroMessage(
                                RobotStatusResponseRecord.class, 
                                null, 
                                RobotStatusResponseRecord.SCHEMA$, 
                                bmsg);
                return new PortableRobotStatusResponse(record);
            }catch(Exception ex){
                theLogger.error("Error deserializing Avro Record.", ex);
                return null;
            }
        }
        
        private RobotResponse getPositionResponse(BytesMessage bmsg){
            try{
                RobotPositionResponseRecord record = 
                        QpidUtils.unpackAvroMessage(
                                RobotPositionResponseRecord.class, 
                                null, 
                                RobotPositionResponseRecord.SCHEMA$, 
                                bmsg);
                return new PortableRobotPositionResponse(record);
            }catch(Exception ex){
                theLogger.error("Error deserializing Avro Record.", ex);
                return null;
            }
        }
        
    }
}
