/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import java.util.logging.Logger;
import javax.jms.Destination;
import javax.jms.Session;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.motion.protocol.MotionFrame;
import org.robokind.api.motion.protocol.MotionFrameEvent;
import org.robokind.avrogen.motion.MotionFrameEventRecord;
import org.robokind.impl.messaging.JMSAvroMessageSender;

/**
 * JMSAvroMessageSender for sending MotionFrameEvents
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSMotionFrameSender extends 
        JMSAvroMessageSender<MotionFrameEvent, MotionFrameEventRecord>{
    static final Logger theLogger = Logger.getLogger(JMSMotionFrameSender.class.getName());
    private String mySourceId;
    private String myDestinationId;
    /**
     * Creates a new JMSMotionFrameSender
     * @param session
     * @param destination 
     */
    public JMSMotionFrameSender(Session session, Destination destination){
        super(session, destination);
        setAdapter(new MotionFrameEventAdapter());
    }
    /**
     * Sets the source id String for this sender.
     * This is used as the sourceId in outgoing MotionFrameEvents.
     * @param sourceId source id String to use
     */
    public void setSourceId(String sourceId){
        mySourceId = sourceId;
    }
    /**
     * Set the destination id String for this sender.
     * This is used as the destinationId in outgoing MotionFrameEvents.
     * @param destId destination id String to use.
     */
    public void setDestinationId(String destId){
        myDestinationId = destId;
    }
    /**
     * Creates a MotionFrameEvent for the given MotionFrame, and sends it.
     * @param frame MotionFrame to send
     * @throws NullPointerException if the sourceId or destinationId for this
     * sender are null, or if the given MotionFrame is null.
     */
    public void sendMotionFrame(MotionFrame frame) {
        if(frame == null || mySourceId == null || myDestinationId == null){
            throw new NullPointerException();
        }
        MotionFrameEvent frameEvent = 
                new PortableMotionFrameEvent(
                        mySourceId, myDestinationId, TimeUtils.now(), frame);
        super.sendMessage(frameEvent);
    }
    /**
     * Converts a MotionFrameEvent to a MotionFrameEventRecord.
     */
    public static class MotionFrameEventAdapter 
            implements Adapter<MotionFrameEvent, MotionFrameEventRecord>{
        @Override
        public MotionFrameEventRecord adapt(MotionFrameEvent a) {
            return new PortableMotionFrameEvent(a).getMotionFrameEventRecord();
        }
    }
}
