/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.jms.Destination;
import javax.jms.Session;
import org.apache.avro.specific.SpecificRecordBase;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.motion.messaging.messages.RobotDefinitionResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotPositionResponse;
import org.robokind.api.motion.messaging.messages.RobotResponse.RobotStatusResponse;
import org.robokind.impl.messaging.JMSAvroMessageSender;
import org.robokind.impl.motion.messaging.command.PortableRobotDefinitionResponse;
import org.robokind.impl.motion.messaging.command.PortableRobotPositionResponse;
import org.robokind.impl.motion.messaging.command.PortableRobotStatusResponse;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSRobotResponseSender extends 
        JMSAvroMessageSender<RobotResponse, SpecificRecordBase> {
    static final Logger theLogger = 
            Logger.getLogger(JMSRobotResponseSender.class.getName());
    
    public JMSRobotResponseSender(Session session, Destination destination){
        super(session, destination);
        setAdapter(new RobotResponseAdapter());
    }
    
    @Override
    public void sendMessage(RobotResponse message){
        theLogger.log(Level.INFO, "Sending Robot Response: {0}, to: {1}", 
                new Object[]{message.getResponseHeader().getRequestType(), 
                    message.getResponseHeader().getRobotId().getRobtIdString()});
        String contentType = getContentType(message);
        if(contentType == null){
            theLogger.warning(
                    "Cannot find contentType, unable to send message.");
            return;
        }
        sendMessage(message, contentType);
    }
    
    public String getContentType(RobotResponse a) {
        if(a == null){
            throw new NullPointerException();
        }
        if(a instanceof RobotDefinitionResponse){
            return JMSRobotResponseBlockingReceiver.MIME_ROBOT_DEFINITION_RESPONSE;
        }else if(a instanceof RobotStatusResponse){
            return JMSRobotResponseBlockingReceiver.MIME_ROBOT_STATUS_RESPONSE;
        }else if(a instanceof RobotPositionResponse){
            return JMSRobotResponseBlockingReceiver.MIME_ROBOT_POSITION_RESPONSE;
        }
        return null;
    }
    /**
     * Converts various RobotResponses to Avro Records.
     * Converts RobotDefinitionResponse to RobotDefinitionResponseRecord
     * Converts RobotStatusResponse to RobotStatusResponseRecord
     * Converts RobotPositionResponse to RobotPositionResponseRecord
     */
    public static class RobotResponseAdapter 
            implements Adapter<RobotResponse, SpecificRecordBase>{
        @Override
        public SpecificRecordBase adapt(RobotResponse a) {
            if(a == null){
                throw new NullPointerException();
            }
            if(a instanceof RobotDefinitionResponse){
                if(a instanceof PortableRobotDefinitionResponse){
                    return ((PortableRobotDefinitionResponse)a).getRecord();
                }
                return new PortableRobotDefinitionResponse(
                        (RobotDefinitionResponse)a).getRecord();
            }else if(a instanceof RobotStatusResponse){
                if(a instanceof PortableRobotStatusResponse){
                    return ((PortableRobotStatusResponse)a).getRecord();
                }
                return new PortableRobotStatusResponse(
                        (RobotStatusResponse)a).getRecord();
            }else if(a instanceof RobotPositionResponse){
                if(a instanceof PortableRobotPositionResponse){
                    return ((PortableRobotPositionResponse)a).getRecord();
                }
                return new PortableRobotPositionResponse(
                        (RobotPositionResponse)a).getRecord();
            }
            return null;
        }
    }
}
