/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging.command;

import org.apache.avro.util.Utf8;
import org.robokind.api.common.position.NormalizedDouble;
import org.robokind.api.motion.Joint;
import org.robokind.api.motion.messaging.messages.RobotDefinitionResponse.JointDefinition;
import org.robokind.avrogen.motion.JointDefinitionRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableJointDefinition implements JointDefinition{
    private JointDefinitionRecord myRecord;
    private Joint.Id myCachedId;
    private NormalizedDouble myCachedDefaultPosition;
    private NormalizedDouble myCachedGoalPosition;
    
    public PortableJointDefinition(JointDefinitionRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        setRecord(record);
    }
    
    public PortableJointDefinition(JointDefinition jointDef){
        if(jointDef == null){
            throw new NullPointerException();
        }else if(jointDef instanceof PortableJointDefinition){
            PortableJointDefinition pr = ((PortableJointDefinition)jointDef);
            setRecord(pr.getRecord());
            return;
        }
        setRecord(jointDef.getJointId(), 
                jointDef.getName().toString(), 
                jointDef.getDefaultPosition(), 
                jointDef.getGoalPosition(), 
                jointDef.getEnabled());
    }
    
    public PortableJointDefinition(Joint.Id jId, String name, 
            NormalizedDouble defPos, NormalizedDouble goalPos, boolean enabled){
        setRecord(jId, name, defPos, goalPos, enabled);
    }
    
    public PortableJointDefinition(Joint joint){
        setRecord(joint.getId(), joint.getName(), 
                joint.getDefaultPosition(), 
                joint.getGoalPosition(), 
                joint.getEnabled());
    }
    
    private void setRecord(Joint.Id jId, String name, 
            NormalizedDouble defPos, NormalizedDouble goalPos, boolean enabled){
        JointDefinitionRecord rec = new JointDefinitionRecord();
        rec.jointId = jId.getLogicalJointNumber();
        rec.name = new Utf8(name);
        rec.defaultPosition = defPos.getValue();
        rec.goalPosition = goalPos.getValue();
        rec.enabled = enabled;
        myCachedId = jId;
        myCachedDefaultPosition = defPos;
        myCachedGoalPosition = goalPos;
        myRecord = rec;
    }
    
    private void setRecord(JointDefinitionRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
        myCachedId = new Joint.Id(myRecord.jointId);
        myCachedDefaultPosition = 
                new NormalizedDouble(myRecord.defaultPosition);
        myCachedGoalPosition = new NormalizedDouble(myRecord.goalPosition);
    }
    
    @Override
    public Joint.Id getJointId() {
        return myCachedId;
    }

    @Override
    public String getName() {
        return myRecord.name.toString();
    }

    @Override
    public NormalizedDouble getDefaultPosition() {
        return myCachedDefaultPosition;
    }

    @Override
    public NormalizedDouble getGoalPosition() {
        return myCachedGoalPosition;
    }

    @Override
    public boolean getEnabled() {
        return myRecord.enabled;
    }
    
    public JointDefinitionRecord getRecord(){
        return myRecord;
    }
}
