/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.impl.motion.osgi;

import java.io.File;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.configuration.HierarchicalConfiguration;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.robokind.impl.motion.config.RobotConfigXMLFileLoader;
import org.robokind.impl.motion.config.RobotConfigXMLReader;
import org.robokind.impl.motion.jointgroup.RobotJointGroupXMLFileLoader;
import org.robokind.impl.motion.jointgroup.RobotJointGroupConfigXMLReader;
import org.robokind.api.motion.jointgroup.RobotJointGroup;
import org.robokind.api.motion.jointgroup.RobotJointGroupFactory;
import org.robokind.api.motion.jointgroup.JointGroup;
import org.robokind.api.common.services.ServiceConfigurationLoader;
import org.robokind.api.common.services.ServiceFactory;
import org.robokind.api.common.services.ServiceUtils;
import org.robokind.api.motion.servos.config.ServoRobotConfig;
import org.robokind.api.motion.jointgroup.RobotJointGroupConfig;
import org.robokind.impl.motion.sync.SynchronizedRobotConfigLoader;
import org.robokind.impl.motion.sync.SynchronizedRobotConfigWriter;

/**
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class Activator implements BundleActivator {
    private final static Logger theLogger = Logger.getLogger(Activator.class.getName());
    
    @Override
    public void start(BundleContext context) throws Exception {
        theLogger.log(Level.INFO, "MotionAPI Activation Begin.");
        registerRobotXMLReader(context);
        registerRobotXMLFileLoader(context);
        registerJointGroupXMLReader(context);
        registerJointGroupXMLFileLoader(context);
        registerRobotJointGroupFactory(context);
        ServiceRegistration loadReg = ServiceUtils.registerConfigurationLoader(
                context, new SynchronizedRobotConfigLoader(), null);
        ServiceRegistration writeReg = ServiceUtils.registerConfigurationWriter(
                context, new SynchronizedRobotConfigWriter(), null);
        theLogger.log(Level.INFO, "MotionAPI Activation Complete.");
    }
    
    private void registerRobotXMLReader(BundleContext context){
        RobotConfigXMLReader reader = new RobotConfigXMLReader(context);
        Properties props = new Properties();
        props.put(ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION, RobotConfigXMLReader.VERSION.toString());
        props.put(ServiceConfigurationLoader.PROP_SERVICE_CONFIG_CLASS, ServoRobotConfig.class.getName());
        props.put(ServiceConfigurationLoader.PROP_PARAMETER_CLASS, HierarchicalConfiguration.class.getName());
        context.registerService(ServiceConfigurationLoader.class.getName(), reader, props);
        theLogger.log(Level.INFO, "RobotConfig XML Reader Service Registered Successfully.");
    }
    
    private void registerRobotXMLFileLoader(BundleContext context){
        RobotConfigXMLFileLoader reader = new RobotConfigXMLFileLoader(context);
        Properties props = new Properties();
        props.put(ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION, RobotConfigXMLReader.VERSION.toString());
        props.put(ServiceConfigurationLoader.PROP_SERVICE_CONFIG_CLASS, ServoRobotConfig.class.getName());
        props.put(ServiceConfigurationLoader.PROP_PARAMETER_CLASS, File.class.getName());
        context.registerService(ServiceConfigurationLoader.class.getName(), reader, props);
        theLogger.log(Level.INFO, "RobotConfig XML File Reader Service Registered Successfully.");
    }
    
    private void registerJointGroupXMLReader(BundleContext context){
        RobotJointGroupConfigXMLReader reader = new RobotJointGroupConfigXMLReader();
        Properties props = new Properties();
        props.put(ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION, RobotJointGroupConfigXMLReader.VERSION.toString());
        props.put(ServiceConfigurationLoader.PROP_SERVICE_CONFIG_CLASS, RobotJointGroupConfig.class.getName());
        props.put(ServiceConfigurationLoader.PROP_PARAMETER_CLASS, HierarchicalConfiguration.class.getName());
        context.registerService(ServiceConfigurationLoader.class.getName(), reader, props);
        theLogger.log(Level.INFO, "JointGroupConfig XML Reader Service Registered Successfully.");
    }
    
    private void registerJointGroupXMLFileLoader(BundleContext context){
        RobotJointGroupXMLFileLoader reader = new RobotJointGroupXMLFileLoader();
        Properties props = new Properties();
        props.put(ServiceConfigurationLoader.PROP_CONFIG_FORMAT_VERSION, RobotJointGroupConfigXMLReader.VERSION.toString());
        props.put(ServiceConfigurationLoader.PROP_SERVICE_CONFIG_CLASS, RobotJointGroupConfig.class.getName());
        props.put(ServiceConfigurationLoader.PROP_PARAMETER_CLASS, File.class.getName());
        context.registerService(ServiceConfigurationLoader.class.getName(), reader, props);
        theLogger.log(Level.INFO, "JointGroupConfig XML File Reader Service Registered Successfully.");
    }
    
    private void registerRobotJointGroupFactory(BundleContext context){
        RobotJointGroupFactory fact = new RobotJointGroupFactory();
        Properties props = new Properties();
        props.put(ServiceFactory.PROP_SERVICE_VERSION, RobotJointGroup.VERSION.toString());
        props.put(ServiceFactory.PROP_SERVICE_CLASS, JointGroup.class.getName());
        context.registerService(ServiceFactory.class.getName(), fact, props);
        theLogger.log(Level.INFO, "RobotJointGroup Factory Service Registered Successfully.");
    }

    @Override
    public void stop(BundleContext context) throws Exception {}

}
