/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.osgi;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.jms.Connection;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Session;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.robokind.api.common.osgi.lifecycle.AbstractLifecycleProvider;
import org.robokind.api.common.osgi.lifecycle.DescriptorListBuilder;
import org.robokind.api.common.osgi.lifecycle.ServiceLifecycleProvider;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.messaging.MoveFrameListener;
import org.robokind.api.motion.messaging.RemoteRobotHost;
import org.robokind.api.motion.utils.RobotMoverFrameSource;
import org.robokind.api.motion.utils.RobotUtils;
import org.robokind.impl.messaging.utils.ConnectionManager;
import org.robokind.impl.motion.messaging.JMSRemoteRobotHost;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class JMSRemoteRobotHostLifecycleProvider extends 
        AbstractLifecycleProvider<RemoteRobotHost, JMSRemoteRobotHost>{
    private final static Logger theLogger = 
            Logger.getLogger(ServiceLifecycleProvider.class.getName());
    private final static String theDepRobotId = "robot";
    private final static String theDepConnectionId = "motionConnection";
    private final static String theDepRequestDestId = "requestDestination";
    private final static String theDepResponseDestId = "responseDestination";
    private final static String theDepMoveDestId = "moveDestination";
    
    private BundleContext myContext;
    private ServiceRegistration myFrameSourceReg;

    public JMSRemoteRobotHostLifecycleProvider(
            BundleContext context, Robot.Id robotId, String connId, 
            String reqDestId, String respDestId, String moveDestId){
        super(new DescriptorListBuilder()
                .addId(theDepRobotId, Robot.class, 
                        Robot.PROP_ID, robotId.getRobtIdString())
                .addId(theDepConnectionId, Connection.class, 
                        ConnectionManager.PROP_CONNECTION_ID, connId)
                .addId(theDepRequestDestId, Destination.class, 
                        ConnectionManager.PROP_DESTINATION_ID, reqDestId)
                .addId(theDepResponseDestId, Destination.class, 
                        ConnectionManager.PROP_DESTINATION_ID, respDestId)
                .addId(theDepMoveDestId, Destination.class, 
                        ConnectionManager.PROP_DESTINATION_ID, moveDestId)
                .getDescriptors());
        if(context == null){
            throw new NullPointerException();
        }
        myContext = context;
    }

    @Override
    protected JMSRemoteRobotHost create(Map<String, Object> services) {
        Robot robot = (Robot)services.get(theDepRobotId);
        Connection con = (Connection)services.get(theDepConnectionId);
        Destination reqDest = (Destination)services.get(theDepRequestDestId);
        Destination respDest = (Destination)services.get(theDepResponseDestId);
        Destination moveDest = (Destination)services.get(theDepMoveDestId);
        Robot.Id id = robot.getRobotId();
        Session session;
        try{
            session = con.createSession(false, Session.CLIENT_ACKNOWLEDGE);
        }catch(JMSException ex){
            theLogger.log(Level.WARNING, "Unable to create Session.", ex);
            return null;
        }
        RobotMoverFrameSource frameSource = new RobotMoverFrameSource(robot);
        MoveFrameListener moveHandler = new MoveFrameListener();
        myFrameSourceReg = 
                RobotUtils.registerFrameSource(myContext, id, frameSource);
        moveHandler.setRobotFrameSource(frameSource);
        try{
            JMSRemoteRobotHost host = new JMSRemoteRobotHost(
                robot, "sourceId", "destId", 
                session, reqDest, respDest, moveDest, moveHandler);
            host.start();
            return host;
        }catch(Exception ex){
            theLogger.log(Level.WARNING, 
                    "There was an error creating and "
                    + "starting the RemoteRobotHost.", ex);
            return null;
        }
    }

    @Override
    protected void handleChange(String serviceId, Object service) {
    }

    @Override
    public Class<RemoteRobotHost> getServiceClass() {
        return RemoteRobotHost.class;
    }
}
