/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.lifecycle;

import java.util.Properties;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import javax.jms.Connection;
import org.osgi.framework.BundleContext;
import org.robokind.api.common.lifecycle.ServiceLifecycleProvider;
import org.robokind.api.common.lifecycle.utils.ManagedServiceGroup;
import org.robokind.api.common.osgi.lifecycle.OSGiComponentFactory;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.lifecycle.RemoteRobotHostLifecycle;
import org.robokind.api.motion.lifecycle.RobotMoveHandlerLifecycle;
import org.robokind.api.motion.protocol.MotionFrameEvent;
import org.robokind.api.motion.protocol.RobotRequest;
import org.robokind.api.motion.protocol.RobotResponse;
import org.robokind.avrogen.motion.MotionFrameEventRecord;
import org.robokind.avrogen.motion.RobotRequestRecord;
import org.robokind.impl.messaging.lifecycle.JMSAvroAsyncReceiverLifecycle;
import org.robokind.impl.messaging.lifecycle.JMSAvroPolymorphicSenderLifecycle;
import org.robokind.impl.messaging.utils.ConnectionUtils;
import org.robokind.impl.motion.messaging.PortableMotionFrameEvent;
import org.robokind.impl.motion.messaging.PortableRobotRequest;
import org.robokind.impl.motion.messaging.PortableRobotResponse;
import static org.robokind.impl.messaging.utils.ConnectionUtils.QUEUE;
/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class RemoteRobotHostServiceGroup extends ManagedServiceGroup{
    private final static Logger theLogger = 
            Logger.getLogger(RemoteRobotHostServiceGroup.class.getName());
    
    private final static String CONNECTION_ID = "motionConnection";
    private final static String REQUEST_DEST_ID = "robotRequest";
    private final static String RESPONSE_DEST_ID = "robotResponse";
    private final static String MOVE_DEST_ID = "robotMotionFrame";
    private final static String REQUEST_DEST = "robotRequest";
    private final static String RESPONSE_DEST = "robotResponse";
    private final static String MOVE_DEST = "motionFrame";
    private final static String REQUEST_RECEIVER_ID = "robotRequestReceiver";
    private final static String RESPONSE_SENDER_ID = "robotResponseSender";
    private final static String MOVE_RECEIVER_ID = "robotFrameReceiver";
    private final static String MOVE_HANDLER_ID = "robotMoveHandler";  
    
    private static String getIdBase(Robot.Id robotId, String serviceId){
        String base = "robot/" + robotId + "/" + serviceId;
        //TODO: sanitize base
        return base;
    }
    private static String getDestBase(Robot.Id robotId, String serviceId){
        String base = "robot" + robotId + serviceId;
        base = base.replaceAll("[^a-zA-Z0-9]+", "");
        return base;
    }
    
    public RemoteRobotHostServiceGroup(BundleContext context,
            Robot.Id robotId, String hostId, String clientId, 
            Connection connection, Properties registrationProperties){
        super(new OSGiComponentFactory(context), 
                getLifecycles(robotId, hostId, clientId), 
                getIdBase(robotId, hostId), 
                registrationProperties);
        String base = getIdBase(robotId, hostId);
        String dbase = getDestBase(robotId, hostId);
        connectJMS(context, 
                id(base, CONNECTION_ID), connection, 
                id(base, REQUEST_DEST_ID), dest(dbase, REQUEST_DEST), 
                id(base, RESPONSE_DEST_ID), dest(dbase, RESPONSE_DEST), 
                id(base, MOVE_DEST_ID), dest(dbase, MOVE_DEST), 
                myServiceProperties);
    }
    
    private static List<ServiceLifecycleProvider> getLifecycles(
            Robot.Id robotId, String hostId, String clientId){
        String base = getIdBase(robotId, hostId);
        return getRemoteRobotHostServices(
                id(base, hostId), 
                id(getIdBase(robotId, clientId), clientId), 
                robotId, 
                id(base, CONNECTION_ID), 
                id(base, REQUEST_DEST_ID), 
                id(base, RESPONSE_DEST_ID), 
                id(base, MOVE_DEST_ID), 
                id(base, REQUEST_RECEIVER_ID), 
                id(base, RESPONSE_SENDER_ID), 
                id(base, MOVE_RECEIVER_ID), 
                id(base, MOVE_HANDLER_ID));
    }
    
    private static List<ServiceLifecycleProvider> getRemoteRobotHostServices(
            String robotHostId, 
            String robotClientId,Robot.Id robotId, 
            String connectionId, String requestDestId, 
            String responseDestId, String moveDestId,
            String requestReceiverId, String responseSenderId, 
            String moveReceiverId, String moveHandlerId){
        List<ServiceLifecycleProvider> lifecycles = new ArrayList();
        lifecycles.add(new JMSAvroPolymorphicSenderLifecycle<RobotResponse>(
                        new PortableRobotResponse.MessageRecordAdapter(), 
                        RobotResponse.class, responseSenderId, 
                        connectionId, responseDestId));
        lifecycles.add(new JMSAvroAsyncReceiverLifecycle(
                        new PortableRobotRequest.RecordMessageAdapter(), 
                        RobotRequest.class, RobotRequestRecord.class, 
                        RobotRequestRecord.SCHEMA$, requestReceiverId, 
                        connectionId, requestDestId));
        lifecycles.add(new JMSAvroAsyncReceiverLifecycle(
                        new PortableMotionFrameEvent.RecordMessageAdapter(), 
                        MotionFrameEvent.class, MotionFrameEventRecord.class, 
                        MotionFrameEventRecord.SCHEMA$, moveReceiverId, 
                        connectionId, moveDestId));
        lifecycles.add(new RobotMoveHandlerLifecycle(moveHandlerId, robotId));
        lifecycles.add(new RemoteRobotHostLifecycle(robotHostId, robotClientId, 
                        robotId, requestReceiverId, responseSenderId, 
                        moveReceiverId, moveHandlerId));
        return lifecycles;
    }
    
    private static void connectJMS(BundleContext context, 
            String connectionId, Connection connection, 
            String requestDestId, String requestDestination,
            String responseDestId, String responseDestination, 
            String moveDestId, String moveDestination,
            Properties registrationProps){
        try{
            theLogger.info("Registering Motion Connection and Destinations");
            ConnectionUtils.ensureSession(context, 
                    connectionId, connection, registrationProps);
            ConnectionUtils.ensureDestinations(context, 
                    requestDestId, requestDestination, QUEUE, registrationProps, 
                    responseDestId, responseDestination, QUEUE, registrationProps,
                    moveDestId, moveDestination, QUEUE, registrationProps);
            theLogger.info("Motion Connection and Destinations Registered");
        }catch(Exception ex){
            
        }
    }
    
    private static String id(String base, String suffix){
        return base + "/" + suffix;
    } 
    
    private static String dest(String base, String suffix){
        String combined = base + suffix;
        combined = combined.replaceAll("[^a-zA-Z0-9]+", "");
        return combined;
    } 
}
