/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import java.util.AbstractMap.SimpleEntry;
import java.util.Map.Entry;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericArray;
import org.apache.avro.generic.GenericData.Array;
import org.apache.avro.util.Utf8;
import org.robokind.api.common.position.NormalizedDouble;
import org.robokind.api.motion.Joint;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.JointId;
import org.robokind.api.motion.Robot.RobotPositionMap;
import org.robokind.api.motion.protocol.DefaultMotionFrame;
import org.robokind.api.motion.protocol.MotionFrame;
import org.robokind.avrogen.motion.JointIdRecord;
import org.robokind.avrogen.motion.JointPositionRecord;
import org.robokind.avrogen.motion.MotionFrameRecord;
import org.robokind.avrogen.motion.RobotPositionMapRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class MotionMessagingUtils {
    public static MotionFrameRecord packMotionFrame(
            MotionFrame<RobotPositionMap> frame){
        MotionFrameRecord frameRec = new MotionFrameRecord();
        frameRec.moveDurationMillisec = frame.getFrameLengthMillisec();
        frameRec.timestampMillisecUTC = frame.getTimestampMillisecUTC();
        RobotPositionMap start = frame.getPreviousPositions();
        RobotPositionMap goal = frame.getGoalPositions();
        if(goal == null){
            throw new NullPointerException();
        }
        frameRec.goalPositions = packRobotPositionMap(goal);
        if(start != null){
            frameRec.startPositions = packRobotPositionMap(start);
        }
        return frameRec;        
    }
    
    public static RobotPositionMapRecord packRobotPositionMap(
            RobotPositionMap map){
        RobotPositionMapRecord mapRec = new RobotPositionMapRecord();
        GenericArray<JointPositionRecord> jointPositions = new Array(map.size(), 
                Schema.createArray(JointPositionRecord.SCHEMA$));
        for(Entry<JointId,NormalizedDouble> e : map.entrySet()){
            JointId jId = e.getKey();
            NormalizedDouble pos = e.getValue();
            JointPositionRecord posRec = packJointPosition(jId, pos);
            jointPositions.add(posRec);
        }
        mapRec.jointPositions = jointPositions;
        return mapRec;
    }
    
    private static JointPositionRecord packJointPosition(
            Robot.JointId jointId, NormalizedDouble position){
        JointIdRecord jointIdRec = packJointId(jointId);
        JointPositionRecord posRec = new JointPositionRecord();
        posRec.jointId = jointIdRec;
        posRec.normalizedPosition = position.getValue();
        return posRec;
    }
    
    public static JointIdRecord packJointId(Robot.JointId jointId){
        JointIdRecord idRec = new JointIdRecord();
        idRec.jointId = jointId.getJointId().getLogicalJointNumber();
        idRec.robotId = new Utf8(jointId.getRobotId().getRobtIdString());
        return idRec;
    }
    
    public static MotionFrame unpackMotionFrame(MotionFrameRecord frameRec){
        MotionFrame<RobotPositionMap> frame = 
                new DefaultMotionFrame<RobotPositionMap>();
        frame.setTimestampMillisecUTC(frameRec.timestampMillisecUTC);
        frame.setFrameLengthMillisec(frameRec.moveDurationMillisec);
        frame.setGoalPositions(unpackPositionMap(frameRec.goalPositions));
        RobotPositionMapRecord startRec = frameRec.startPositions;
        if(startRec != null){
            frame.setPreviousPositions(unpackPositionMap(startRec));
        }
        return frame;
    }
    
    public static RobotPositionMap unpackPositionMap(
            RobotPositionMapRecord mapRec){
        GenericArray<JointPositionRecord> posRecs = mapRec.jointPositions;
        RobotPositionMap map = new Robot.RobotPositionHashMap(posRecs.size());
        for(JointPositionRecord posRec : posRecs){
            Entry<JointId,NormalizedDouble> e = unpackJointPosition(posRec);
            map.put(e.getKey(), e.getValue());
        }
        return map;
    }
    
    private static Entry<JointId,NormalizedDouble> unpackJointPosition(
            JointPositionRecord posRec){
        JointId jId = unpackJointId(posRec.jointId);
        NormalizedDouble pos = new NormalizedDouble(posRec.normalizedPosition);
        Entry<JointId,NormalizedDouble> e = 
                new SimpleEntry<JointId, NormalizedDouble>(jId,pos);
        return e;
    }
    
    public static JointId unpackJointId(JointIdRecord jointIdRec){
        Robot.Id rId = new Robot.Id(jointIdRec.robotId.toString());
        Joint.Id jId = new Joint.Id(jointIdRec.jointId);
        JointId jointId = new JointId(rId, jId);
        return jointId;
    }
}
