/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.motion.Robot;
import org.robokind.api.motion.Robot.Id;
import org.robokind.api.motion.Robot.JointId;
import org.robokind.api.motion.messaging.RobotRequestFactory;
import org.robokind.api.motion.protocol.RobotRequest;
import org.robokind.avrogen.motion.RobotRequestRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableRobotRequest implements RobotRequest{
    private RobotRequestRecord myRecord;
    private Robot.Id myCachedId;
    
    public PortableRobotRequest(RobotRequestRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
        myCachedId = new Robot.Id(myRecord.robotId.toString());
    }
    
    public PortableRobotRequest(RobotRequest request){
        if(request == null){
            throw new NullPointerException();
        }else if(request instanceof PortableRobotRequest){
            myRecord = ((PortableRobotRequest)request).getRecord();
            myCachedId = new Robot.Id(myRecord.robotId.toString());
            return;
        }
        myCachedId = request.getRobotId();
        myRecord = new RobotRequestRecord();
        myRecord.robotId = new Utf8(myCachedId.getRobtIdString());
        myRecord.requestSourceId = new Utf8(request.getSourceId());
        myRecord.requestDestinationId = new Utf8(request.getDestinationId());
        myRecord.requestType = new Utf8(request.getRequestType());
        myRecord.timestampMillisecUTC = request.getTimestampMillisecUTC();
    }
    
    public PortableRobotRequest(
            Robot.Id robotId, String sourceId, String destId, 
            String requestType, long timestampMillisecUTC,
            Integer requestIndex){
        myCachedId = robotId;
        myRecord = new RobotRequestRecord();
        myRecord.robotId = new Utf8(myCachedId.getRobtIdString());
        myRecord.requestSourceId = new Utf8(sourceId);
        myRecord.requestDestinationId = new Utf8(destId);
        myRecord.requestType = new Utf8(requestType);
        myRecord.timestampMillisecUTC = timestampMillisecUTC;
        myRecord.intParam = requestIndex;
    }
    
    @Override
    public Id getRobotId() {
        return myCachedId;
    }

    @Override
    public String getSourceId() {
        return myRecord.requestSourceId.toString();
    }

    @Override
    public String getDestinationId() {
        return myRecord.requestDestinationId.toString();
    }

    @Override
    public String getRequestType() {
        return myRecord.requestType.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
       return myRecord.timestampMillisecUTC;
    }

    @Override
    public Integer getRequestIndex() {
        return myRecord.intParam;
    }
    
    public RobotRequestRecord getRecord(){
        return myRecord;
    }
    
        public static class Factory implements 
                RobotRequestFactory<PortableRobotRequest> {

        @Override
        public PortableRobotRequest buildRobotRequest(Robot.Id robotId, 
                String sourceId, String destId, 
                String requestType, long timestampMillisecUTC) {
            return new PortableRobotRequest(robotId, sourceId, destId, 
                    requestType, timestampMillisecUTC, null);
        }

        @Override
        public PortableRobotRequest buildJointRequest(
                JointId jointId, String sourceId, String destId, 
                String requestType, long timestampMillisecUTC) {
            Robot.Id robotId = jointId.getRobotId();
            Integer jointIdInt = jointId.getJointId().getLogicalJointNumber();
            return new PortableRobotRequest(robotId, sourceId, destId, 
                    requestType, timestampMillisecUTC, jointIdInt);
        }

    }
    
    public static class MessageRecordAdapter implements 
            Adapter<RobotRequest,RobotRequestRecord>{

        @Override
        public RobotRequestRecord adapt(RobotRequest a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableRobotRequest(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<RobotRequestRecord, RobotRequest>{
        @Override
        public RobotRequest adapt(RobotRequestRecord a) {
            return new PortableRobotRequest(a);
        }
    }
    
}
