/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.motion.messaging;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.common.position.NormalizableRange;
import org.robokind.api.motion.JointProperty;
import org.robokind.api.motion.protocol.RobotDefinitionResponse.JointPropDefinition;
import org.robokind.avrogen.motion.JointPropDefinitionRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */


public class PortableJointPropDefinition  implements JointPropDefinition{
    private JointPropDefinitionRecord myRecord;
    
    public PortableJointPropDefinition(JointPropDefinitionRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableJointPropDefinition(JointPropDefinition def){
        if(def == null){
            throw new NullPointerException();
        }else if(def instanceof PortableJointDefinition){
            PortableJointPropDefinition pr = ((PortableJointPropDefinition)def);
            myRecord = pr.getRecord();
            return;
        }
        setRecord(def.getPropertyName(), def.getDisplayName(), 
                def.getMinValue(), def.getMaxValue(), def.getInitialValue());
    }
    
    public PortableJointPropDefinition(
            String propName, String displayName,
            double minVal, double maxVal, double initialVal){
        setRecord(propName, displayName, minVal, maxVal, initialVal);
    }
    
    public PortableJointPropDefinition(JointProperty prop){
        Class c = prop.getPropertyClass();
        NormalizableRange r = prop.getNormalizableRange();
        if(!Number.class.isAssignableFrom(c)){
            setRecord(prop.getPropertyName(), prop.getDisplayName(), 
                    0.0, 1.0, r.normalizeValue(prop.getValue()).getValue());
            return;
        }
        Number n = (Number)prop.getValue();
        Number min = (Number)r.getMin();
        Number max = (Number)r.getMax();
        setRecord(prop.getPropertyName(), prop.getDisplayName(), 
                min.doubleValue(), max.doubleValue(), n.doubleValue());
    }
    
    private void setRecord(
            String propName, String displayName,
            double minVal, double maxVal, double initialVal){
        JointPropDefinitionRecord rec = new JointPropDefinitionRecord();
        rec.propertyName = new Utf8(propName);
        rec.displayName = new Utf8(displayName);
        rec.minValue = minVal;
        rec.maxValue = maxVal;
        rec.initialValue = initialVal;
        myRecord = rec;
    }
    
    @Override
    public String getPropertyName() {
        return myRecord.propertyName.toString();
    }

    @Override
    public String getDisplayName() {
        return myRecord.displayName.toString();
    }

    @Override
    public Double getMinValue() {
        return myRecord.minValue;
    }

    @Override
    public Double getMaxValue() {
        return myRecord.maxValue;
    }

    @Override
    public Double getInitialValue() {
        return myRecord.initialValue;
    }
    
    public JointPropDefinitionRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<JointPropDefinition,JointPropDefinitionRecord>{

        @Override
        public JointPropDefinitionRecord adapt(JointPropDefinition a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableJointPropDefinition(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<JointPropDefinitionRecord, JointPropDefinition>{
        @Override
        public JointPropDefinition adapt(JointPropDefinitionRecord a) {
            return new PortableJointPropDefinition(a);
        }
    }
}
