/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.speech.SpeechConfig;
import org.robokind.avrogen.speech.SpeechConfigRecord;
import org.robokind.impl.messaging.services.PortableServiceCommand;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableSpeechConfig implements SpeechConfig{
    private SpeechConfigRecord myRecord;
    
    public PortableSpeechConfig(SpeechConfigRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableSpeechConfig(SpeechConfig config){
        if(config == null){
            throw new NullPointerException();
        }else if(config instanceof PortableServiceCommand){
            myRecord = ((PortableSpeechConfig)config).getRecord();
            return;
        }
        setRecord(config.getSpeechServiceId(), config.getConfigSourceId(), 
                config.getVoiceName(), config.getSampleRate());
    }
    
    public PortableSpeechConfig(String speechServiceId, String configSourceId, 
            String voiceName, double sampleRate){
        setRecord(speechServiceId, configSourceId, voiceName, sampleRate);
    }
    private void setRecord(String speechServiceId, String configSourceId, 
            String voiceName, double sampleRate){
        if(speechServiceId == null || 
                configSourceId == null || 
                voiceName == null){
            throw new NullPointerException();
        }
        myRecord = new SpeechConfigRecord();
        myRecord.speechServiceId = new Utf8(speechServiceId);
        myRecord.configSourceId = new Utf8(configSourceId);
        myRecord.voiceName = new Utf8(voiceName);
        myRecord.sampleRate = sampleRate;
    }
    
    @Override
    public String getSpeechServiceId() {
        return myRecord.speechServiceId.toString();
    }

    @Override
    public String getConfigSourceId() {
        return myRecord.configSourceId.toString();
    }
    
    @Override
    public String getVoiceName() {
        return myRecord.voiceName.toString();
    }

    @Override
    public double getSampleRate() {
        return myRecord.sampleRate;
    }
    
    public SpeechConfigRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechConfig,SpeechConfigRecord>{

        @Override
        public SpeechConfigRecord adapt(SpeechConfig a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechConfig(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<SpeechConfigRecord, SpeechConfig>{

        @Override
        public SpeechConfig adapt(SpeechConfigRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechConfig(a);
        }
        
    }
}
