/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.speech.SpeechEvent;
import org.robokind.avrogen.speech.SpeechEventRecord;

/**
 * SpeechEvent implementation wrapping a SpeechEventRecord.
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableSpeechEvent implements SpeechEvent {
    
    private SpeechEventRecord myRecord;
    long myTimestampUTC;
    
    /**
     * Creates a new SpeechEvent from the given SpeechEventRecord.
     * @param record SpeechEventRecord to wrap
     */
    public PortableSpeechEvent(SpeechEventRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableSpeechEvent(SpeechEvent event){
        if(event == null){
            throw new NullPointerException();
        }else if(event instanceof PortableSpeechEvent){
            myRecord = ((PortableSpeechEvent)event).getRecord();
            return;
        }
        setRecord(event.getSpeechEventType(), event.getStreamNumber(), 
                event.getTextPosition(), event.getTextLength(), 
                event.getCurrentData(), event.getNextData(), 
                event.getStringData(), event.getDuration());
    }
    
    public PortableSpeechEvent(String speechEventType, long streamNumber, 
            int textPosition, int textLength, 
            int currentData, int nextData, 
            String stringData, int duration){
        setRecord(speechEventType, streamNumber, textPosition, textLength, 
                currentData, nextData, stringData, duration);
    }
    
    private void setRecord(String speechEventType, long streamNumber, 
            int textPosition, int textLength, 
            int currentData, int nextData, 
            String stringData, int duration){
        myRecord = new SpeechEventRecord();
        myRecord.eventType = new Utf8(speechEventType);
        myRecord.streamNumber = streamNumber;
        myRecord.textPosition = textPosition;
        myRecord.textLength = textLength;
        myRecord.currentData = currentData;
        myRecord.nextData = nextData;
        myRecord.stringData = new Utf8(stringData);
        myRecord.duration = duration;
    }

    @Override
    public String getSpeechEventType() {
        return myRecord.eventType.toString();
    }

    @Override
    public Long getStreamNumber() {
        return myRecord.streamNumber;
    }

    @Override
    public Integer getTextPosition() {
        return myRecord.textPosition;
    }

    @Override
    public Integer getTextLength() {
        return myRecord.textLength;
    }

    @Override
    public Integer getCurrentData() {
        return myRecord.currentData;
    }

    @Override
    public Integer getNextData() {
        return myRecord.nextData;
    }

    @Override
    public String getStringData() {
        return myRecord.stringData.toString();
    }

    @Override
    public Integer getDuration() {
        return myRecord.duration;
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myTimestampUTC;
    }

    public void setTimestampMillisecUTC(long timestamp) {
        myTimestampUTC = timestamp;
    }
    
    public SpeechEventRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechEvent,SpeechEventRecord>{

        @Override
        public SpeechEventRecord adapt(SpeechEvent a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechEvent(a).getRecord();
        }
    }
    
    /**
     * Converts a SpeechEventRecord to a PortableSpeechEvent.
     */
    public static class RecordMessageAdapter 
            implements Adapter<SpeechEventRecord, SpeechEvent>{

        @Override
        public SpeechEvent adapt(SpeechEventRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechEvent(a);
        }
        
    }
}
