/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech;

import java.util.ArrayList;
import java.util.List;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData.Array;
import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.speech.SpeechEvent;
import org.robokind.api.speech.SpeechEventList;
import org.robokind.avrogen.speech.SpeechEventListRecord;
import org.robokind.avrogen.speech.SpeechEventRecord;

/**
 * SpeechEventList implementation wrapping a SpeechEventListRecord.
 * Converts the SpeechEventsRecords to a List of PortableSpeechEvents.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableSpeechEventList implements SpeechEventList{
    private SpeechEventListRecord myListRecord;
    private List<SpeechEvent> myEvents;
    
    /**
     * Creates a new PortableSpeechEventList from the given Record
     * @param record SpeechEventListRecord to wrap
     */
    public PortableSpeechEventList(SpeechEventListRecord record){
        setRecord(record);
    }
    
    public PortableSpeechEventList(SpeechEventList eventList){
        if(eventList == null){
            throw new NullPointerException();
        }
        if(eventList instanceof PortableSpeechEventList){
            setRecord(((PortableSpeechEventList)eventList).getRecord());
        }
        myListRecord = new SpeechEventListRecord();
        List<SpeechEvent> events = eventList.getSpeechEvents();
        myListRecord.speechEvents = 
                new Array<SpeechEventRecord>(events.size(), 
                        Schema.createArray(SpeechEventListRecord.SCHEMA$));
        myEvents = new ArrayList<SpeechEvent>(events.size());
        for(SpeechEvent event : events){
            PortableSpeechEvent e = new PortableSpeechEvent(event);
            e.setTimestampMillisecUTC(eventList.getTimestampMillisecUTC());
            myEvents.add(event);
            myListRecord.speechEvents.add(e.getRecord());
        }
        myListRecord.speechServiceId = new Utf8(eventList.getSpeechServiceId());
        myListRecord.timestampMillisecUTC = eventList.getTimestampMillisecUTC();
    }
    
    private void setRecord(SpeechEventListRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myListRecord = record;
        myEvents = new ArrayList<SpeechEvent>(myListRecord.speechEvents.size());
        Adapter<SpeechEventRecord,SpeechEvent> adapter = 
                new PortableSpeechEvent.RecordMessageAdapter();
        for(SpeechEventRecord rec : myListRecord.speechEvents){
            SpeechEvent ev = adapter.adapt(rec);
            if(ev instanceof PortableSpeechEvent){
                ((PortableSpeechEvent)ev).setTimestampMillisecUTC(
                        record.timestampMillisecUTC);
            }
            myEvents.add(ev);
        }
    }
    
    
    @Override
    public String getSpeechServiceId() {
        return myListRecord.speechServiceId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myListRecord.timestampMillisecUTC;
    }

    @Override
    public List<SpeechEvent> getSpeechEvents() {
        return myEvents;
    }
    
    public SpeechEventListRecord getRecord(){
        return myListRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechEventList,SpeechEventListRecord>{

        @Override
        public SpeechEventListRecord adapt(SpeechEventList a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechEventList(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<SpeechEventListRecord, SpeechEventList>{

        @Override
        public SpeechEventList adapt(SpeechEventListRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechEventList(a);
        }
    }
}
