/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.speech.SpeechRequest;
import org.robokind.api.speech.SpeechRequestFactory;
import org.robokind.avrogen.speech.SpeechRequestRecord;

/**
 * SpeechRequest implementation wrapping a SpeechRequestRecord.
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableSpeechRequest implements 
        SpeechRequest {
    private SpeechRequestRecord myRequestRecord;
    
    /**
     * Creates a new PortableSpeechRequest from the given Record.
     * @param record SpeechRequestRecord to wrap
     */
    public PortableSpeechRequest(SpeechRequestRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRequestRecord = record;
    }
    /**
     * Creates a new PortableSpeechRequest.
     * @param speechServiceId id String of the SpeechService
     * @param requestSourceId id String of the SpeechRequest source
     * @param timestampMillisecUTC speech request timestamp
     * @param phrase 
     */
    public PortableSpeechRequest(
            String speechServiceId, String requestSourceId, 
            long timestampMillisecUTC, String phrase){
        if(speechServiceId == null ||
                requestSourceId == null ||
                phrase == null){
            throw new NullPointerException();
        }        
        myRequestRecord = new SpeechRequestRecord();
        myRequestRecord.speechServiceId = new Utf8(speechServiceId);
        myRequestRecord.requestSourceId = new Utf8(requestSourceId);
        myRequestRecord.timestampMillisecUTC = timestampMillisecUTC;
        myRequestRecord.phrase = new Utf8(phrase);
    }
    
    /**
     * Creates a PortableSpeechRequest from another SpeechRequest.
     * If the given SpeechRequest is a PortableSpeechRequest, the Record is
     * simply copied.  Otherwise, a new REcord is created and populated from the
     * given request.
     * @param req SpeechRequest to copy
     */
    public PortableSpeechRequest(SpeechRequest req){
        if(req instanceof PortableSpeechRequest){
            myRequestRecord = ((PortableSpeechRequest)req).getRecord();
            return;
        }
        if(req.getSpeechServiceId() == null ||
                req.getRequestSourceId() == null ||
                req.getPhrase() == null){
            throw new NullPointerException();
        }        
        myRequestRecord = new SpeechRequestRecord();
        myRequestRecord.speechServiceId = new Utf8(req.getSpeechServiceId());
        myRequestRecord.requestSourceId = new Utf8(req.getRequestSourceId());
        myRequestRecord.timestampMillisecUTC = req.getTimestampMillisecUTC();
        myRequestRecord.phrase = new Utf8(req.getPhrase());
    }
    
    /**
     * Returns the underlying SpeechRequestRecord.
     * @return underlying SpeechRequestRecord
     */
    public SpeechRequestRecord getRecord(){
        return myRequestRecord;
    }
    
    @Override
    public String getSpeechServiceId() {
        return myRequestRecord.speechServiceId.toString();
    }

    @Override
    public String getRequestSourceId() {
        return myRequestRecord.requestSourceId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myRequestRecord.timestampMillisecUTC;
    }

    @Override
    public String getPhrase() {
        return myRequestRecord.phrase.toString();
    }
    
    public static class Factory implements SpeechRequestFactory{
        @Override
        public SpeechRequest create(
                String clientId, String hostId, String phrase) {
            return new PortableSpeechRequest(
                    hostId, clientId, TimeUtils.now(), phrase);
        }
    }
    
    public static class MessageRecordAdapter 
            implements Adapter<SpeechRequest, SpeechRequestRecord>{
        @Override
        public SpeechRequestRecord adapt(SpeechRequest a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRequest(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter 
            implements Adapter<SpeechRequestRecord, SpeechRequest>{
        @Override
        public SpeechRequest adapt(SpeechRequestRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRequest(a);
        }
    }
    
}
