/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech.viseme;

import java.util.EnumMap;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData.Array;
import org.robokind.api.common.position.NormalizedDouble;
import org.robokind.api.common.utils.Utils;
import org.robokind.api.speech.viseme.Viseme;
import org.robokind.api.speech.viseme.config.VisemeBindingConfig;
import org.robokind.avrogen.speech.viseme.VisemeBindingConfigRecord;
import org.robokind.avrogen.speech.viseme.VisemePositionRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableVisemeBindingConfig implements VisemeBindingConfig{
    private VisemeBindingConfigRecord myRecord;
    private Map<Viseme,NormalizedDouble> myCachedPositions;
    
    public PortableVisemeBindingConfig(VisemeBindingConfigRecord record){
        myRecord = record;
        myCachedPositions = new EnumMap<Viseme, NormalizedDouble>(Viseme.class);
        for(VisemePositionRecord pos : record.visemeBindings){
            int id = pos.visemeId;
            Viseme vis = Viseme.getById(id);
            double boundPos = Utils.bound(pos.position, 0.0, 1.0);
            NormalizedDouble val = new NormalizedDouble(boundPos);
            myCachedPositions.put(vis, val);
        }
    }
    
    public PortableVisemeBindingConfig(VisemeBindingConfig conf){
        if(conf instanceof PortableVisemeBindingConfig){
            myRecord = ((PortableVisemeBindingConfig)conf).getRecord();
            myCachedPositions = conf.getVisemeValues();
            return;
        }
        init(conf.getBindingKey(), conf.getVisemeValues());
        
    }
    
    public PortableVisemeBindingConfig(int bindingId, Map<Viseme,NormalizedDouble> visPos){
        init(bindingId, visPos);
    }
    
    private void init(int bindingId, Map<Viseme,NormalizedDouble> visPos){
        myRecord = new VisemeBindingConfigRecord();
        myRecord.bindingId = bindingId;
        myRecord.visemeBindings = new Array<VisemePositionRecord>(
                visPos.size(), 
                Schema.createArray(VisemePositionRecord.SCHEMA$));
        for(Entry<Viseme,NormalizedDouble> e : visPos.entrySet()){
            VisemePositionRecord posRec = new VisemePositionRecord();
            posRec.visemeId = e.getKey().getVisemeId();
            posRec.position = e.getValue().getValue();
            myRecord.visemeBindings.add(posRec);
        }
        myCachedPositions = new EnumMap<Viseme, NormalizedDouble>(visPos);
    }
    
    @Override
    public int getBindingKey() {
        return myRecord.bindingId;
    }

    @Override
    public Map<Viseme, NormalizedDouble> getVisemeValues() {
        return myCachedPositions;
    }
    
    public VisemeBindingConfigRecord getRecord(){
        return myRecord;
    }
    
}
