/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speech;

import java.util.Properties;
import java.util.logging.Logger;
import org.jflux.api.core.Listener;
import org.jflux.api.core.config.Configuration;
import org.osgi.framework.BundleContext;
import org.robokind.api.common.lifecycle.utils.ManagedServiceFactory;
import org.robokind.api.common.osgi.lifecycle.OSGiComponent;
import org.robokind.api.common.osgi.lifecycle.OSGiComponentFactory;
import org.robokind.api.messaging.services.ServiceCommand;
import org.robokind.api.messaging.services.ServiceError;
import org.robokind.api.speech.SpeechConfig;
import org.robokind.api.speech.SpeechEventList;
import org.robokind.api.speech.SpeechRequest;
import org.robokind.api.speech.lifecycle.RemoteSpeechServiceClientLifecycle;
import org.robokind.api.speech.viseme.lifecycle.VisemeEventNotifierLifecycle;
import org.robokind.impl.messaging.config.DependentLifecycle;
import org.robokind.impl.messaging.config.MessagingLifecycleGroupConfigUtils;
import org.robokind.impl.messaging.config.RKMessagingConfigUtils;
import static org.robokind.impl.messaging.config.MessagingLifecycleGroupConfigUtils.*;
import static org.robokind.api.common.lifecycle.config.RKManagedGroupConfigUtils.*;

/**
 *
 * @author Matthew Stevenson
 */
public class RemoteSpeechUtils {
    private final static Logger theLogger = 
            Logger.getLogger(RemoteSpeechUtils.class.getName());
    
    private final static String COMMAND_DEST_CONFIG_ID = "speechCommandDestConfig";
    private final static String CONFIG_DEST_CONFIG_ID = "speechConfigDestConfig";
    private final static String ERROR_DEST_CONFIG_ID = "speechErrorDestConfig";
    private final static String REQUEST_DEST_CONFIG_ID = "speechRequestDestConfig";
    private final static String EVENT_DEST_CONFIG_ID = "speechEventDestConfig";
    
    private final static String COMMAND_SERIALIZE_CONFIG_ID = ServiceCommand.class.toString();
    private final static String CONFIG_SERIALIZE_CONFIG_ID = SpeechConfig.class.toString();
    private final static String ERROR_SERIALIZE_CONFIG_ID = ServiceError.class.toString();
    private final static String REQUEST_SERIALIZE_CONFIG_ID = SpeechRequest.class.toString();
    private final static String EVENT_SERIALIZE_CONFIG_ID = SpeechEventList.class.toString();
    
    private final static String COMMAND_DEST_NAME = "speechCommand";
    private final static String CONFIG_DEST_NAME = "speechCommand";
    private final static String ERROR_DEST_NAME = "speechError";
    private final static String REQUEST_DEST_NAME = "speechRequest";
    private final static String EVENT_DEST_NAME = "speechEvent";
    
    private final static String COMMAND_SENDER_ID = "speechCommand";
    private final static String CONFIG_SENDER_ID = "speechConfig";
    private final static String ERROR_RECEIVER_ID = "speechError";
    private final static String REQUEST_SENDER_ID = "speechRequest";
    private final static String EVENT_RECEIVER_ID = "speechEvent";
        
    public final static String GROUP_PREFIX = "RKSpeechGroup";
    
    public static void connect(BundleContext context, 
            String speechGroupId, String connectionConfigId) {
        if(context == null 
                || speechGroupId ==  null || connectionConfigId == null){
            throw new NullPointerException();
        }
        ManagedServiceFactory fact = new OSGiComponentFactory(context);
        registerDestConfigs(fact);
        launchComponents(speechGroupId, connectionConfigId, null, fact);
        
        new OSGiComponent(context,
                new VisemeEventNotifierLifecycle(speechGroupId)).start();
        launchRemoteSpeechClient(context, speechGroupId, 
                speechGroupId,  COMMAND_SENDER_ID, CONFIG_SENDER_ID, 
                ERROR_RECEIVER_ID, REQUEST_SENDER_ID, EVENT_RECEIVER_ID);
    }
    
    private static void registerDestConfigs(ManagedServiceFactory fact){
        RKMessagingConfigUtils.registerQueueConfig(
                COMMAND_DEST_CONFIG_ID, COMMAND_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerQueueConfig(
                CONFIG_DEST_CONFIG_ID, CONFIG_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerTopicConfig(
                ERROR_DEST_CONFIG_ID, ERROR_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerQueueConfig(
                REQUEST_DEST_CONFIG_ID, REQUEST_DEST_NAME,  null, fact);
        RKMessagingConfigUtils.registerTopicConfig(
                EVENT_DEST_CONFIG_ID, EVENT_DEST_NAME,  null, fact);
    }
    
    private static void launchComponents(
            String groupId, String connectionConfigId,
            Properties props, ManagedServiceFactory fact){
        groupId += "/" + GROUP_PREFIX;
        launchComponent(groupId + "/" + COMMAND_SENDER_ID, props, REMOTE_NOTIFIER, 
                COMMAND_DEST_CONFIG_ID, connectionConfigId,
                COMMAND_SERIALIZE_CONFIG_ID, fact);
        launchComponent(groupId + "/" + CONFIG_SENDER_ID, props, REMOTE_NOTIFIER, 
                CONFIG_DEST_CONFIG_ID, connectionConfigId,
                CONFIG_SERIALIZE_CONFIG_ID, fact);
        launchComponent(groupId + "/" + ERROR_RECEIVER_ID, props, REMOTE_LISTENER, 
                ERROR_DEST_CONFIG_ID, connectionConfigId, 
                ERROR_SERIALIZE_CONFIG_ID, fact);
        launchComponent(groupId + "/" + REQUEST_SENDER_ID, props, REMOTE_NOTIFIER, 
                REQUEST_DEST_CONFIG_ID, connectionConfigId, 
                REQUEST_SERIALIZE_CONFIG_ID, fact);
        launchComponent(groupId + "/" + EVENT_RECEIVER_ID, props, REMOTE_LISTENER, 
                EVENT_DEST_CONFIG_ID, connectionConfigId, 
                EVENT_SERIALIZE_CONFIG_ID, fact);
    }
    
    private static String launchComponent(
            final String groupId, 
            final Properties props, 
            final int componentType, 
            final String destinationConfigId, 
            final String connectionConfigId,
            String serializeConfigId, 
            ManagedServiceFactory fact){
        final ManagedGroupFactory mgf = new ManagedGroupFactory(fact); 
        DependentLifecycle.createDependencyListener(
                RKMessagingConfigUtils.SERIALIZATION_CONFIG, 
                serializeConfigId, Configuration.class, 
                new Listener<Configuration<String>>() {
                    @Override
                    public void handleEvent(Configuration<String> event) {
                        mgf.adapt(buildMessagingComponentLifecycleGroupConfig(
                                groupId, props, componentType, event, 
                                destinationConfigId, connectionConfigId));
                    }
                }, fact);
        return groupId(groupId, groupId, groupId);
    }
    
    private static void launchRemoteSpeechClient(
            BundleContext context,
            String speechClientId, String speechHostId,
            String commandSenderId, String configSenderId, 
            String errorReceiverId, String speechRequestSenderId,
            String speechEventsReceiverId){
        String idBase = speechClientId + "/" + GROUP_PREFIX;
        RemoteSpeechServiceClientLifecycle lifecycle =
                new RemoteSpeechServiceClientLifecycle(
                        speechClientId, speechHostId, 
                        groupId(idBase, commandSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, configSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, errorReceiverId, LISTENER_COMPONENT), 
                        groupId(idBase, speechRequestSenderId, NOTIFIER_COMPONENT), 
                        groupId(idBase, speechEventsReceiverId, LISTENER_COMPONENT));
        OSGiComponent speechComp = new OSGiComponent(context, lifecycle);
        speechComp.start();
    }
    private static String groupId(String groupId, String suffix, String component){
        return MessagingLifecycleGroupConfigUtils.childId(groupId + "/" + suffix, component);   
    }    
}
