/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speechrec;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.speechrec.SpeechRecEvent;
import org.robokind.avrogen.speechrec.SpeechRecEventRecord;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableSpeechRecEvent implements SpeechRecEvent{
    private SpeechRecEventRecord myRecord;
    
    public PortableSpeechRecEvent(SpeechRecEventRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableSpeechRecEvent(SpeechRecEvent event){
        if(event == null){
            throw new NullPointerException();
        }else if(event instanceof PortableSpeechRecEvent){
            myRecord = ((PortableSpeechRecEvent)event).getRecord();
            return;
        }
        setRecord(
                event.getRecognizerId(), 
                event.getRecognizedText(), 
                event.getConfidence(),
                event.getTimestampMillisecUTC());
    }
    
    public PortableSpeechRecEvent(
            String recognizerId, String phrase, double conf, long timestamp){
        setRecord(recognizerId, phrase, conf, timestamp);
    }
    
    private void setRecord(
            String recognizerId, String phrase, double conf, long timestamp){
        myRecord = new SpeechRecEventRecord();
        myRecord.timestampMillisecUTC = TimeUtils.now();
        myRecord.recognizerId = new Utf8(recognizerId);
        myRecord.recognizedText = new Utf8(phrase);
        myRecord.confidence = conf;
    }
    
    @Override
    public String getRecognizerId() {
        return myRecord.recognizerId.toString();
    }

    @Override
    public long getTimestampMillisecUTC() {
        return myRecord.timestampMillisecUTC;
    }

    @Override
    public String getRecognizedText() {
        return myRecord.recognizedText.toString();
    }

    @Override
    public double getConfidence() {
        return myRecord.confidence;
    }
    
    public SpeechRecEventRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechRecEvent,SpeechRecEventRecord>{
        @Override
        public SpeechRecEventRecord adapt(SpeechRecEvent a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecEvent(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<SpeechRecEventRecord, SpeechRecEvent>{

        @Override
        public SpeechRecEvent adapt(SpeechRecEventRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecEvent(a);
        }
    }
}
