/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speechrec;

import java.util.ArrayList;
import java.util.List;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.speechrec.SpeechRecEvent;
import org.robokind.api.speechrec.SpeechRecEventList;
import org.robokind.avrogen.speechrec.SpeechRecEventRecord;
import org.robokind.avrogen.speechrec.SpeechRecEventListRecord;

/**
 *
 * @author Jason G. Pallack <www.robokind.org>
 */
public class PortableSpeechRecEventList implements SpeechRecEventList{
    private SpeechRecEventListRecord myRecord;
    private List<SpeechRecEvent> myEvents;
    
    public PortableSpeechRecEventList(SpeechRecEventListRecord record){
        setRecord(record);
    }
    
    public PortableSpeechRecEventList(SpeechRecEventList eventList){
        if(eventList == null){
            throw new NullPointerException();
        }
        if(eventList instanceof PortableSpeechRecEventList){
            setRecord(((PortableSpeechRecEventList)eventList).getRecord());
        }
        myRecord = new SpeechRecEventListRecord();
        List<SpeechRecEvent> events = eventList.getSpeechRecEvents();
        myRecord.speechRecEvents = 
                new GenericData.Array<SpeechRecEventRecord>(events.size(), 
                        Schema.createArray(SpeechRecEventListRecord.SCHEMA$));
        myEvents = new ArrayList<SpeechRecEvent>(events.size());
        for(SpeechRecEvent event : events){
            PortableSpeechRecEvent e = new PortableSpeechRecEvent(event);
            myEvents.add(event);
            myRecord.speechRecEvents.add(e.getRecord());
        }
        myRecord.speechRecServiceId = new Utf8(eventList.getSpeechRecServiceId());
        myRecord.timestampMillisecUTC = eventList.getTimestampMillisecUTC();
        myRecord.eventDestinationId = new Utf8(eventList.getEventDestinationId());
    }
    
    public PortableSpeechRecEventList(
            String sourceId, String destId, 
            List<SpeechRecEvent> events, long timestamp){
        if(sourceId == null || destId == null || events == null){
            throw new NullPointerException();
        }
        myRecord = new SpeechRecEventListRecord();
        myRecord.speechRecEvents = 
                new GenericData.Array<SpeechRecEventRecord>(events.size(), 
                        Schema.createArray(SpeechRecEventListRecord.SCHEMA$));
        myEvents = new ArrayList<SpeechRecEvent>(events.size());
        for(SpeechRecEvent event : events){
            PortableSpeechRecEvent e = new PortableSpeechRecEvent(event);
            myEvents.add(event);
            myRecord.speechRecEvents.add(e.getRecord());
        }
        myRecord.speechRecServiceId = new Utf8(sourceId);
        myRecord.timestampMillisecUTC = timestamp;
        myRecord.eventDestinationId = new Utf8(destId);
    }
      
    private void setRecord(SpeechRecEventListRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
        myEvents = new ArrayList<SpeechRecEvent>(myRecord.speechRecEvents.size());
        Adapter<SpeechRecEventRecord,SpeechRecEvent> adapter = 
                new PortableSpeechRecEvent.RecordMessageAdapter();
        for(SpeechRecEventRecord rec : myRecord.speechRecEvents){
            myEvents.add(adapter.adapt(rec));
        }
    }
    
    @Override
    public long getTimestampMillisecUTC() {
        return myRecord.timestampMillisecUTC;
    }
    
    @Override
    public String getSpeechRecServiceId() {
        return myRecord.speechRecServiceId.toString();
    }
    
    @Override
    public String getEventDestinationId() {
        return myRecord.eventDestinationId.toString();
    }
    
    @Override
    public List<SpeechRecEvent> getSpeechRecEvents() {
        return myEvents;
    }
    
    public SpeechRecEventListRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechRecEventList,SpeechRecEventListRecord>{
        @Override
        public SpeechRecEventListRecord adapt(SpeechRecEventList a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecEventList(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<SpeechRecEventListRecord, SpeechRecEventList>{

        @Override
        public SpeechRecEventList adapt(SpeechRecEventListRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecEventList(a);
        }
    }
}
