/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.speechrec;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.common.utils.TimeUtils;
import org.robokind.api.speechrec.SpeechRecConfig;
import org.robokind.api.speechrec.utils.SpeechRecConfigFactory;
import org.robokind.avrogen.speechrec.SpeechRecConfigRecord;
import org.robokind.impl.messaging.services.PortableServiceCommand;

/**
 *
 * @author Jason G. Pallack <www.robokind.org>
 */
public class PortableSpeechRecConfig implements SpeechRecConfig{
    private SpeechRecConfigRecord myRecord;
    
    public PortableSpeechRecConfig(SpeechRecConfigRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableSpeechRecConfig(SpeechRecConfig config){
        if(config == null){
            throw new NullPointerException();
        }else if(config instanceof PortableServiceCommand){
            myRecord = ((PortableSpeechRecConfig)config).getRecord();
            return;
        }
        setRecord(config.getSpeechRecServiceId(), config.getConfigSourceId(), 
                config.getTimestampMillisecUTC(), config.getVocabLocation());
    }
    
    public PortableSpeechRecConfig(String speechRecServiceId,
            String configSourceId, long timestampMillisecUTC,
            String vocabLocation){
        setRecord(speechRecServiceId, configSourceId, timestampMillisecUTC,
                vocabLocation);
    }
    private void setRecord(String speechRecServiceId, String configSourceId, 
            long timestampMillisecUTC, String vocabLocation){
        if(speechRecServiceId == null || 
                configSourceId == null || 
                vocabLocation == null){
            throw new NullPointerException();
        }
        myRecord = new SpeechRecConfigRecord();
        myRecord.speechRecServiceId = new Utf8(speechRecServiceId);
        myRecord.configSourceId = new Utf8(configSourceId);
        myRecord.timestampMillisecUTC = timestampMillisecUTC;
        myRecord.vocabLocation = new Utf8(vocabLocation);
    }
    
    @Override
    public String getSpeechRecServiceId() {
        return myRecord.speechRecServiceId.toString();
    }

    @Override
    public String getConfigSourceId() {
        return myRecord.configSourceId.toString();
    }
    
    @Override
    public long getTimestampMillisecUTC() {
        return myRecord.timestampMillisecUTC;
    }
    
    @Override
    public String getVocabLocation() {
        return myRecord.vocabLocation.toString();
    }
   
    public SpeechRecConfigRecord getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<SpeechRecConfig,SpeechRecConfigRecord>{

        @Override
        public SpeechRecConfigRecord adapt(SpeechRecConfig a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecConfig(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<SpeechRecConfigRecord, SpeechRecConfig>{

        @Override
        public SpeechRecConfig adapt(SpeechRecConfigRecord a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableSpeechRecConfig(a);
        }   
    }
    
    public static class Factory implements SpeechRecConfigFactory{
        @Override
        public SpeechRecConfig create(
                String speechRecServiceId, String configSourceId,
                String vocabLocation) {
            return new PortableSpeechRecConfig(
                    speechRecServiceId, configSourceId, TimeUtils.now(),
                    vocabLocation);
        }
    }
}
