/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.vision;

import org.apache.avro.generic.GenericArray;
import org.robokind.api.common.utils.Adapter;
import org.robokind.api.vision.ImageRegion;
import org.robokind.api.vision.ImageRegionList;
import org.robokind.avrogen.vision.ImageRegionListRecord;
import org.robokind.avrogen.vision.ImageRegionRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * ImageRegionList implementation wrapping an ImageRegionListRecord.
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableImageRegionList implements ImageRegionList{
    private final static Logger theLogger = LoggerFactory.getLogger(PortableImageRegionList.class);
    private ImageRegionListRecord myImageRegionListRecord;
    private PortableImageRegion[] myRegionArray;

    /**
     * Creates a new PortableImageRegionList from an ImageRegionListRecord.
     * @param regions ImageRegionListRecord to wrap
     */
    public PortableImageRegionList(ImageRegionListRecord regions){
        if(regions == null){
            throw new NullPointerException();
        }
        myImageRegionListRecord = regions;
        GenericArray<ImageRegionRecord> regionArr = regions.regions;
        if(regionArr == null){
            myRegionArray = new PortableImageRegion[0];
        }
        int size = regionArr.size();
        myRegionArray = new PortableImageRegion[regionArr.size()];
        for(int i=0; i<size; i++){
            try{
                ImageRegionRecord region = regionArr.get(i);
                if(region == null){
                    continue;
                }
                myRegionArray[i] = new PortableImageRegion(region);
            }catch(IndexOutOfBoundsException ex){
                theLogger.warn("Index out of bounds.", ex);
            }
        }
    }
    /**
     * Creates a PortableImageRegionList from an existing ImageRegionList.
     * @param regionList ImageRegionList to wrap
     */
    public PortableImageRegionList(ImageRegionList regionList){
        if(regionList == null){
            throw new NullPointerException();
        }
        if(regionList instanceof PortableImageRegionList){
            PortableImageRegionList plist = (PortableImageRegionList)regionList;
            myImageRegionListRecord = plist.myImageRegionListRecord;
            myRegionArray = plist.myRegionArray;
        }
        //TODO Finish me
    }
    
    @Override
    public String getImageSourceId() {
        return myImageRegionListRecord.imageSourceId.toString();
    }

    @Override
    public long getImageId() {
        return myImageRegionListRecord.imageId;
    }

    @Override
    public long getImageTimestampUTC() {
        return myImageRegionListRecord.imageTimestampMillisecUTC;
    }

    @Override
    public String getImageProcessorId() {
        return myImageRegionListRecord.imageProcessorId.toString();
    }

    @Override
    public long getImageRegionsId() {
        return myImageRegionListRecord.imageRegionsId;
    }

    @Override
    public long getProcessorStartTimestampUTC() {
        return myImageRegionListRecord.processorStartTimestampMillisecUTC;
    }

    @Override
    public long getProcessorCompleteTimestampUTC() {
        return myImageRegionListRecord.processorCompleteTimestampMillisecUTC;
    }

    @Override
    public ImageRegion[] getImageRegions() {
        return myRegionArray;
    }
    /**
     * Returns the underlying ImageRegionListRecord.
     * @return underlying ImageRegionListRecord
     */
    public ImageRegionListRecord getImageRegionListRecord(){
        return myImageRegionListRecord;
    }
    /**
     * Converts an ImageRegionListRecor to a PortableImageRegionList
     */
    public static class ImageRegionListRecordAdapter implements Adapter<ImageRegionListRecord,ImageRegionList>{
        @Override
        public ImageRegionList adapt(ImageRegionListRecord a) {
            return new PortableImageRegionList(a);
        }
    }
}
