/*
 * Copyright 2012 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.robokind.impl.vision;

import org.apache.avro.util.Utf8;
import org.jflux.api.core.util.Adapter;
import org.robokind.api.vision.config.FaceDetectServiceConfig;
import org.robokind.avrogen.vision.FaceDetectConfig;

/**
 *
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableFaceDetectServiceConfig implements FaceDetectServiceConfig{
    private FaceDetectConfig myRecord;
    
    public PortableFaceDetectServiceConfig(FaceDetectConfig record){
        if(record == null){
            throw new NullPointerException();
        }
        myRecord = record;
    }
    
    public PortableFaceDetectServiceConfig(FaceDetectServiceConfig config){
        if(config == null){
            throw new NullPointerException();
        }
        setRecord(config.getConfigSourceId(), config.getFaceDetectServiceId(), 
                config.getHaarFileLocation(), config.getScaleFactor(), 
                config.getNeighbors(), config.getHaarFlags(), 
                config.getMinFaceWidth(), config.getMinFaceHeight());
    }
    
    public PortableFaceDetectServiceConfig(String sourceId, 
            String faceDetectServiceId, String haarLocation, 
            double scaleFactor, int neighbors, int haarFlags, 
            int minFaceWidth, int minFaceHeight){
        setRecord(sourceId, faceDetectServiceId, 
                haarLocation, scaleFactor, neighbors, 
                haarFlags, minFaceWidth, minFaceHeight);
    }
    
    private void setRecord(String sourceId, 
            String faceDetectServiceId, String haarLocation, 
            double scaleFactor, int neighbors, int haarFlags, 
            int minFaceWidth, int minFaceHeight){
        myRecord = new FaceDetectConfig();
        myRecord.brokerAddress = new Utf8("");
        myRecord.connectionOptions = new Utf8("");
        myRecord.haarFileLocation = new Utf8(haarLocation);
        myRecord.haarFlags = haarFlags;
        myRecord.minFaceHeight = minFaceHeight;
        myRecord.minFaceWidth = minFaceWidth;
        myRecord.neighbors = neighbors;
        myRecord.receivingAddress = new Utf8("");
        myRecord.resultsAddress = new Utf8("");
        myRecord.scaleFactor = scaleFactor;
    }
    
    @Override
    public String getConfigSourceId() {
        return myRecord.resultsAddress.toString();
    }

    @Override
    public String getFaceDetectServiceId() {
        return myRecord.resultsAddress.toString();
    }

    @Override
    public String getHaarFileLocation() {
        return myRecord.haarFileLocation.toString();
    }

    @Override
    public double getScaleFactor() {
        return myRecord.scaleFactor;
    }

    @Override
    public int getNeighbors() {
        return myRecord.neighbors;
    }

    @Override
    public int getHaarFlags() {
        return myRecord.haarFlags;
    }

    @Override
    public int getMinFaceWidth() {
        return myRecord.minFaceWidth;
    }

    @Override
    public int getMinFaceHeight() {
        return myRecord.minFaceHeight;
    }
    
    public FaceDetectConfig getRecord(){
        return myRecord;
    }
    
    public static class MessageRecordAdapter implements 
            Adapter<FaceDetectServiceConfig,FaceDetectConfig>{

        @Override
        public FaceDetectConfig adapt(FaceDetectServiceConfig a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableFaceDetectServiceConfig(a).getRecord();
        }
    }
    
    public static class RecordMessageAdapter implements
            Adapter<FaceDetectConfig, FaceDetectServiceConfig>{

        @Override
        public FaceDetectServiceConfig adapt(FaceDetectConfig a) {
            if(a == null){
                throw new NullPointerException();
            }
            return new PortableFaceDetectServiceConfig(a);
        }
        
    }
}
