/*
 * Copyright 2011 Hanson Robokind LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.robokind.impl.vision;

import java.awt.Image;
import java.nio.ByteBuffer;
import org.apache.avro.util.Utf8;
import org.jflux.api.core.Adapter;
import org.robokind.api.vision.ImageEvent;
import org.robokind.avrogen.vision.ImageRecord;

/**
 * ImageEvent implementation wrapping an ImageEventRecord.
 * Creates and caches a Java Image from the image data.
 * 
 * @author Matthew Stevenson <www.robokind.org>
 */
public class PortableImageEvent implements ImageEvent{
    private ImageRecord myImageRecord;
    private Image myCachedImage;
    
    /**
     * Creates a new PortableImageEvent wrapping the given ImageRecord.
     * @param record ImageRecord to wrap
     */
    public PortableImageEvent(ImageRecord record){
        if(record == null){
            throw new NullPointerException();
        }
        myImageRecord = record;
    }

    @Override
    public Image getImage() {
        if(myCachedImage == null){
            myCachedImage = PortableImageUtils.unpackImage(myImageRecord);
        }
        return myCachedImage;
    }
    
    /**
     * Returns the underlying ImageRecord.
     * @return underlying ImageRecord
     */
    public ImageRecord getImageRecord(){
        return myImageRecord;
    }

    @Override
    public String getImageSourceId() {
        return myImageRecord.imageSourceId.toString();
    }

    @Override
    public long getImageTimestampUTC() {
        return myImageRecord.imageTimestampMillisecUTC;
    }

    @Override
    public long getImageId() {
        return myImageRecord.imageId;
    }

    @Override
    public int getWidth() {
        return myImageRecord.width;
    }

    @Override
    public int getHeight() {
        return myImageRecord.height;
    }

    @Override
    public int getChannelCount() {
        return myImageRecord.nChannels;
    }

    @Override
    public int getWidthStep() {
        return myImageRecord.widthStep;
    }

    @Override
    public ByteBuffer getImageData() {
        return myImageRecord.data;
    }
    /**
     * Adapts Avro ImageRecords to ImageEvents.
     */
    public static class RecordMessageAdapter 
            implements Adapter<ImageRecord,ImageEvent>{
        @Override
        public ImageEvent adapt(ImageRecord a) {
            return new PortableImageEvent(a);
        }
    }
    public static class MessageRecordAdapter 
            implements Adapter<ImageEvent,ImageRecord>{
        @Override
        public ImageRecord adapt(ImageEvent a) {
            ImageRecord rec = new ImageRecord();
            rec.data = a.getImageData();
            rec.height = a.getHeight();
            rec.imageId = a.getImageId();
            rec.imageSourceId = new Utf8(a.getImageSourceId());
            rec.imageTimestampMillisecUTC = a.getImageTimestampUTC();
            rec.nChannels = a.getChannelCount();
            rec.width = a.getWidth();
            rec.widthStep = a.getWidthStep();
            return rec;
        }
    }
}
