/*
 * Copyright 2016 SimplifyOps, Inc. (http://simplifyops.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
* PoliciesCache.java
* 
* User: Greg Schueler <a href="mailto:greg@dtosolutions.com">greg@dtosolutions.com</a>
* Created: Nov 16, 2010 11:26:12 AM
* 
*/
package com.dtolabs.rundeck.core.authorization.providers;

import com.dtolabs.rundeck.core.authorization.Attribute;
import com.dtolabs.rundeck.core.authorization.ValidationSet;
import org.yaml.snakeyaml.parser.ParserException;

import java.io.File;
import java.util.*;

/**
 * PoliciesCache retains PolicyDocument objects for inserted Files, and reloads them if file modification time changes.
 *
 * @author Greg Schueler <a href="mailto:greg@dtosolutions.com">greg@dtosolutions.com</a>
 */
public class PoliciesCache implements Iterable<PolicyCollection> {
    static final long DIR_LIST_CHECK_DELAY = Long.getLong(PoliciesCache.class.getName()+".DirListCheckDelay", 60000);
    static final long FILE_CHECK_DELAY = Long.getLong(PoliciesCache.class.getName() + ".FileCheckDelay", 60000);

    private Set<File> warned = new HashSet<File>();
    private Map<String, CacheItem> cache = new HashMap<>();
    private SourceProvider provider;
    /**
     * Context to load the polices within, invalid policies will be flagged
     */
    final private Set<Attribute> forcedContext;
    private Logger logger;

    private PoliciesCache(final SourceProvider provider) {
        this(provider, null, null);
    }
    private PoliciesCache(final SourceProvider provider, final Set<Attribute> forcedContext) {
        this(provider, forcedContext, null);
    }
    private PoliciesCache(final SourceProvider provider, final Set<Attribute> forcedContext, Logger logger) {
        this.provider = provider;
        this.forcedContext = forcedContext;
        this.logger=logger;
    }

    private static class CacheItem{
        PolicyCollection policyCollection;
        Long cacheTime;
        Long modTime;

        private CacheItem(PolicyCollection policyCollection, Long modTime) {
            this.policyCollection = policyCollection;
            this.modTime = modTime;
            this.cacheTime=System.currentTimeMillis();
        }

        public void touch(Long time) {
            this.cacheTime = time;
        }
    }

    private PolicyCollection createEntry(final YamlSource source, final ValidationSet validation) throws PoliciesParseException {
        try {
            return YamlProvider.policiesFromSource(source, forcedContext, validation);
        } catch (ParserException e1) {
            throw new PoliciesParseException("YAML syntax error: " + e1.toString(), e1);
        }catch (Exception e1) {
            throw new PoliciesParseException(e1);
        }
    }

    /**
     * @param source source
     * @return collection
     * @throws PoliciesParseException
     */
    public synchronized PolicyCollection getDocument(final CacheableYamlSource source) throws PoliciesParseException {
//        cacheTotal++;
        CacheItem entry = cache.get(source.getIdentity());

        long checkTime = System.currentTimeMillis();
        if (null == entry || ((checkTime - entry.cacheTime) > FILE_CHECK_DELAY)) {
            final long lastmod = source.getLastModified().getTime();
            if (null == entry || lastmod > entry.modTime) {
                    if (!source.isValid()) {
                        CacheItem remove = cache.remove(source.getIdentity());
                        entry = null;
//                        cacheRemove++;
                    } else {
//                        cacheMiss++;
                        ValidationSet validation = new ValidationSet();
                        PolicyCollection entry1 = createEntry(source, validation);
                        validation.complete();
                        if (null != entry1) {
                            entry = new CacheItem(entry1, lastmod);
                            cache.put(source.getIdentity(), entry);

                            if(!validation.isValid()){
                                warn(validation.toString());
                            }
                        } else {
                            cache.remove(source.getIdentity());
                            entry = null;

                            if(!validation.isValid()){
                                throw new PoliciesParseException(validation.toString());
                            }
                        }
                    }
            }else{
//                cacheUnmodifiedHit++;
                entry.touch(checkTime);
            }
        }else{
//            cacheHit++;
        }
        return null != entry ? entry.policyCollection : null;
    }

    public Iterator<PolicyCollection> iterator() {
        return new cacheIterator(provider.getSourceIterator());
    }

    /**
     * Create a cache from a single file source
     * @param singleFile file
     * @return cache
     */
    public static PoliciesCache fromFile(File singleFile) {
        return fromSourceProvider(YamlProvider.getFileProvider(singleFile));
    }

    /**
     * Create a cache from a single file source
     *
     * @param singleFile file
     *
     * @return cache
     */
    public static PoliciesCache fromFile(File singleFile, Set<Attribute> forcedContext) {
        return fromSourceProvider(YamlProvider.getFileProvider(singleFile), forcedContext);
    }


    /**
     * Create from a provider
     * @param provider source provider
     * @return policies cache
     */
    public static PoliciesCache fromSourceProvider(final SourceProvider provider) {
        return fromSourceProvider(provider, (Logger) null);
    }
    public static PoliciesCache fromSourceProvider(final SourceProvider provider, Logger logger) {
        return new PoliciesCache(provider, null, logger);
    }

    /**
     * Create from a provider with a forced context
     * @param provider source provider
     * @param forcedContext forced context
     * @return policies cache
     */
    public static PoliciesCache fromSourceProvider(
            final SourceProvider provider,
            final Set<Attribute> forcedContext
    )
    {
        return fromSourceProvider(provider, forcedContext, null);
    }/**
     * Create from a provider with a forced context
     * @param provider source provider
     * @param forcedContext forced context
     * @return policies cache
     */
    public static PoliciesCache fromSourceProvider(
            final SourceProvider provider,
            final Set<Attribute> forcedContext,
            final Logger logger
    )
    {
        return new PoliciesCache(provider, forcedContext, logger);
    }

    /**
     * Create a cache from a directory source
     * @param rootDir base director
     * @return cache
     */
    public static PoliciesCache fromDir(File rootDir) {
        return fromSourceProvider(YamlProvider.getDirProvider(rootDir));
    }

    /**
     * Create a cache from a directory source
     * @param rootDir base director
     * @return cache
     */
    public static PoliciesCache fromDir(File rootDir, final Set<Attribute> forcedContext, Logger logger) {
        return fromSourceProvider(YamlProvider.getDirProvider(rootDir), forcedContext, logger);
    }
    /**
     * Create a cache from a directory source
     * @param rootDir base director
     * @return cache
     */
    public static PoliciesCache fromDir(File rootDir, final Set<Attribute> forcedContext) {
        return fromSourceProvider(YamlProvider.getDirProvider(rootDir),forcedContext);
    }
    /**
     * Create a cache from cacheable sources
     * @param sources source
     * @return cache
     */
    public static PoliciesCache fromSources(final Iterable<CacheableYamlSource> sources) {
        return fromSourceProvider(
                new SourceProvider() {
                    @Override
                    public Iterator<CacheableYamlSource> getSourceIterator() {
                        return sources.iterator();
                    }
                }
        );
    }
    /**
     * Create a cache from cacheable sources
     * @param sources source
     * @return cache
     */
    public static PoliciesCache fromSources(final Iterable<CacheableYamlSource> sources, final Set<Attribute> context) {
        return fromSourceProvider(
                new SourceProvider() {
                    @Override
                    public Iterator<CacheableYamlSource> getSourceIterator() {
                        return sources.iterator();
                    }
                },
                context
        );
    }


    private Map<CacheableYamlSource, Long> cooldownset = Collections.synchronizedMap(new HashMap<CacheableYamlSource, Long>());
    /**
     * Iterator over the PolicyCollections for the cache's sources.  It skips
     * sources that are no longer valid
     */
    private class cacheIterator implements Iterator<PolicyCollection> {
        Iterator<CacheableYamlSource> intIter;
        private CacheableYamlSource nextSource;
        private PolicyCollection nextPolicyCollection;

        public cacheIterator(final Iterator<CacheableYamlSource> intIter) {
            this.intIter = intIter;
            nextSource = this.intIter.hasNext() ? this.intIter.next() : null;
            loadNextSource();
        }

        private void loadNextSource() {
            while (hasNextFile() && null == nextPolicyCollection) {
                CacheableYamlSource newNextSource = getNextSource();
                Long aLong = cooldownset.get(newNextSource);
                if (null != aLong && newNextSource.getLastModified().getTime() == aLong) {
                    debug("Skip parsing of: " + newNextSource + ". Reason: parse error cooldown until modified");
                    continue;
                } else if (null != aLong) {
                    //clear
                    cooldownset.remove(newNextSource);
                }
                try {
                    nextPolicyCollection = getDocument(newNextSource);
                } catch (PoliciesParseException e) {
                    error("ERROR unable to parse aclpolicy: " + newNextSource + ". Reason: " + e.getMessage());
                    debug("ERROR unable to parse aclpolicy: " + newNextSource + ". Reason: " + e.getMessage(), e);
                    cache.remove(newNextSource.getIdentity());
                    cooldownset.put(newNextSource, newNextSource.getLastModified().getTime());
                }
            }
        }

        private CacheableYamlSource getNextSource() {
            CacheableYamlSource next = nextSource;
            nextSource = intIter.hasNext() ? intIter.next() : null;
            return next;
        }

        private PolicyCollection getNextPolicyCollection() {
            PolicyCollection doc = nextPolicyCollection;
            nextPolicyCollection =null;
            loadNextSource();
            return doc;
        }

        public boolean hasNextFile() {
            return null != nextSource;
        }

        public boolean hasNext() {
            return null != nextPolicyCollection;
        }

        public PolicyCollection next() {
            return getNextPolicyCollection();
        }

        public void remove() {
        }
    }

    private void warn(final String log) {
        if(null!=logger) {
            logger.warn(log);
        }
    }

    private void debug(final String log, final PoliciesParseException e) {
        if(null!=logger) {
            logger.debug(log, e);
        }
    }

    private void error(final String log) {
        if(null!=logger) {
            logger.error(log);
        }
    }

    private void debug(final String log) {
        if(null!=logger) {
            logger.debug(log);
        }
    }

}
