/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common;

import java.nio.ByteBuffer;
import java.util.List;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.MessageTooLargeException;
import org.eclipse.jetty.websocket.api.ProtocolException;
import org.eclipse.jetty.websocket.api.WebSocketBehavior;
import org.eclipse.jetty.websocket.api.WebSocketException;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.extensions.Extension;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.api.extensions.IncomingFrames;
import org.eclipse.jetty.websocket.common.CloseInfo;
import org.eclipse.jetty.websocket.common.OpCode;
import org.eclipse.jetty.websocket.common.WebSocketFrame;
import org.eclipse.jetty.websocket.common.io.payload.CloseReasonValidator;
import org.eclipse.jetty.websocket.common.io.payload.DeMaskProcessor;
import org.eclipse.jetty.websocket.common.io.payload.NoOpValidator;
import org.eclipse.jetty.websocket.common.io.payload.PayloadProcessor;
import org.eclipse.jetty.websocket.common.io.payload.UTF8Validator;

public class Parser {
    private static final Logger LOG = Log.getLogger(Parser.class);
    private final WebSocketPolicy policy;
    private final ByteBufferPool bufferPool;
    private State state = State.START;
    private int cursor = 0;
    private WebSocketFrame frame;
    private Frame priorDataFrame;
    private byte lastDataOpcode;
    private ByteBuffer payload;
    private int payloadLength;
    private PayloadProcessor maskProcessor = new DeMaskProcessor();
    private PayloadProcessor strictnessProcessor;
    private boolean rsv1InUse = false;
    private boolean rsv2InUse = false;
    private boolean rsv3InUse = false;
    private boolean isTextFrameValidated = true;
    private IncomingFrames incomingFramesHandler;

    public Parser(WebSocketPolicy wspolicy, ByteBufferPool bufferPool) {
        this.bufferPool = bufferPool;
        this.policy = wspolicy;
    }

    private void assertSanePayloadLength(long len) {
        LOG.debug("Payload Length: " + len, new Object[0]);
        if (len > Integer.MAX_VALUE) {
            throw new MessageTooLargeException("[int-sane!] cannot handle payload lengths larger than 2147483647");
        }
        this.policy.assertValidMessageSize((int)len);
        switch (this.frame.getOpCode()) {
            case 8: {
                if (len == 1L) {
                    throw new ProtocolException("Invalid close frame payload length, [" + this.payloadLength + "]");
                }
            }
            case 9: 
            case 10: {
                if (len <= 125L) break;
                throw new ProtocolException("Invalid control frame payload length, [" + this.payloadLength + "] cannot exceed [" + 125 + "]");
            }
        }
    }

    public void configureFromExtensions(List<? extends Extension> exts) {
        this.rsv1InUse = false;
        this.rsv2InUse = false;
        this.rsv3InUse = false;
        this.isTextFrameValidated = true;
        for (Extension extension : exts) {
            if (extension.isRsv1User()) {
                this.rsv1InUse = true;
            }
            if (extension.isRsv2User()) {
                this.rsv2InUse = true;
            }
            if (extension.isRsv3User()) {
                this.rsv3InUse = true;
            }
            if (!extension.isTextDataDecoder()) continue;
            this.isTextFrameValidated = false;
        }
    }

    public IncomingFrames getIncomingFramesHandler() {
        return this.incomingFramesHandler;
    }

    public WebSocketPolicy getPolicy() {
        return this.policy;
    }

    public boolean isRsv1InUse() {
        return this.rsv1InUse;
    }

    public boolean isRsv2InUse() {
        return this.rsv2InUse;
    }

    public boolean isRsv3InUse() {
        return this.rsv3InUse;
    }

    protected void notifyFrame(Frame f) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} Notify {}", new Object[]{this.policy.getBehavior(), this.incomingFramesHandler});
        }
        if (this.policy.getBehavior() == WebSocketBehavior.SERVER && !f.isMasked()) {
            throw new ProtocolException("Client frames MUST be masked (RFC-6455)");
        }
        if (this.incomingFramesHandler == null) {
            return;
        }
        try {
            this.incomingFramesHandler.incomingFrame(f);
        }
        catch (WebSocketException e) {
            this.notifyWebSocketException(e);
        }
        catch (Throwable t) {
            LOG.warn(t);
            this.notifyWebSocketException(new WebSocketException(t));
        }
    }

    protected void notifyWebSocketException(WebSocketException e) {
        LOG.warn(e);
        if (this.incomingFramesHandler == null) {
            return;
        }
        this.incomingFramesHandler.incomingError(e);
    }

    public synchronized void parse(ByteBuffer buffer) {
        if (buffer.remaining() <= 0) {
            return;
        }
        try {
            while (this.parseFrame(buffer)) {
                LOG.debug("{} Parsed Frame: {}", new Object[]{this.policy.getBehavior(), this.frame});
                this.notifyFrame(this.frame);
                if (this.frame.isDataFrame() && this.frame.isFin()) {
                    this.priorDataFrame = null;
                    continue;
                }
                this.priorDataFrame = this.frame;
            }
        }
        catch (WebSocketException e) {
            buffer.position(buffer.limit());
            this.payload = null;
            this.notifyWebSocketException(e);
        }
        catch (Throwable t) {
            buffer.position(buffer.limit());
            this.payload = null;
            this.notifyWebSocketException(new WebSocketException(t));
        }
    }

    private boolean parseFrame(ByteBuffer buffer) {
        if (buffer.remaining() <= 0) {
            return false;
        }
        LOG.debug("{} Parsing {} bytes", new Object[]{this.policy.getBehavior(), buffer.remaining()});
        while (buffer.hasRemaining()) {
            switch (this.state) {
                case START: {
                    if (this.frame != null && this.frame.isFin()) {
                        this.frame.reset();
                    }
                    this.state = State.FINOP;
                    break;
                }
                case FINOP: {
                    byte opc;
                    byte b = buffer.get();
                    boolean fin = (b & 0x80) != 0;
                    boolean rsv1 = (b & 0x40) != 0;
                    boolean rsv2 = (b & 0x20) != 0;
                    boolean rsv3 = (b & 0x10) != 0;
                    byte opcode = opc = (byte)(b & 0xF);
                    if (!OpCode.isKnown(opcode)) {
                        throw new ProtocolException("Unknown opcode: " + opc);
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("OpCode {}, fin={} rsv={}{}{}", OpCode.name(opcode), fin, Character.valueOf(rsv1 ? (char)'1' : '.'), Character.valueOf(rsv2 ? (char)'1' : '.'), Character.valueOf(rsv3 ? (char)'1' : '.'));
                    }
                    if (!this.rsv1InUse && rsv1) {
                        throw new ProtocolException("RSV1 not allowed to be set");
                    }
                    if (!this.rsv2InUse && rsv2) {
                        throw new ProtocolException("RSV2 not allowed to be set");
                    }
                    if (!this.rsv3InUse && rsv3) {
                        throw new ProtocolException("RSV3 not allowed to be set");
                    }
                    boolean isContinuation = false;
                    switch (opcode) {
                        case 1: {
                            if (this.isTextFrameValidated) {
                                this.strictnessProcessor = new UTF8Validator();
                                break;
                            }
                            this.strictnessProcessor = NoOpValidator.INSTANCE;
                            break;
                        }
                        case 8: {
                            this.strictnessProcessor = new CloseReasonValidator();
                            break;
                        }
                        default: {
                            this.strictnessProcessor = NoOpValidator.INSTANCE;
                        }
                    }
                    if (OpCode.isControlFrame(opcode)) {
                        if (!fin) {
                            throw new ProtocolException("Fragmented Control Frame [" + OpCode.name(opcode) + "]");
                        }
                    } else if (opcode == 0) {
                        isContinuation = true;
                        if (this.priorDataFrame == null) {
                            throw new ProtocolException("CONTINUATION frame without prior !FIN");
                        }
                        opcode = this.lastDataOpcode;
                    } else if (OpCode.isDataFrame(opcode) && this.priorDataFrame != null && !this.priorDataFrame.isFin()) {
                        throw new ProtocolException("Unexpected " + OpCode.name(opcode) + " frame, was expecting CONTINUATION");
                    }
                    this.frame = new WebSocketFrame(opcode);
                    this.frame.setFin(fin);
                    this.frame.setRsv1(rsv1);
                    this.frame.setRsv2(rsv2);
                    this.frame.setRsv3(rsv3);
                    this.frame.setContinuation(isContinuation);
                    if (this.frame.isDataFrame()) {
                        this.lastDataOpcode = opcode;
                    }
                    this.state = State.PAYLOAD_LEN;
                    break;
                }
                case PAYLOAD_LEN: {
                    byte b = buffer.get();
                    this.frame.setMasked((b & 0x80) != 0);
                    this.payloadLength = (byte)(0x7F & b);
                    if (this.payloadLength == 127) {
                        this.payloadLength = 0;
                        this.state = State.PAYLOAD_LEN_BYTES;
                        this.cursor = 8;
                        break;
                    }
                    if (this.payloadLength == 126) {
                        this.payloadLength = 0;
                        this.state = State.PAYLOAD_LEN_BYTES;
                        this.cursor = 2;
                        break;
                    }
                    this.assertSanePayloadLength(this.payloadLength);
                    if (this.frame.isMasked()) {
                        this.state = State.MASK;
                        break;
                    }
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case PAYLOAD_LEN_BYTES: {
                    byte b = buffer.get();
                    --this.cursor;
                    this.payloadLength |= (b & 0xFF) << 8 * this.cursor;
                    if (this.cursor != 0) break;
                    this.assertSanePayloadLength(this.payloadLength);
                    if (this.frame.isMasked()) {
                        this.state = State.MASK;
                        break;
                    }
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case MASK: {
                    byte[] m = new byte[4];
                    this.frame.setMask(m);
                    if (buffer.remaining() >= 4) {
                        buffer.get(m, 0, 4);
                        if (this.payloadLength == 0) {
                            this.state = State.START;
                            return true;
                        }
                        this.maskProcessor.reset(this.frame);
                        this.state = State.PAYLOAD;
                        break;
                    }
                    this.state = State.MASK_BYTES;
                    this.cursor = 4;
                    break;
                }
                case MASK_BYTES: {
                    byte b;
                    this.frame.getMask()[4 - this.cursor] = b = buffer.get();
                    --this.cursor;
                    if (this.cursor != 0) break;
                    if (this.payloadLength == 0) {
                        this.state = State.START;
                        return true;
                    }
                    this.maskProcessor.reset(this.frame);
                    this.state = State.PAYLOAD;
                    break;
                }
                case PAYLOAD: {
                    if (!this.parsePayload(buffer)) break;
                    if (this.frame.getOpCode() == 8) {
                        new CloseInfo(this.frame);
                    }
                    this.state = State.START;
                    return true;
                }
            }
        }
        return false;
    }

    private boolean parsePayload(ByteBuffer buffer) {
        if (this.payloadLength == 0) {
            return true;
        }
        while (buffer.hasRemaining()) {
            if (this.payload == null) {
                this.frame.assertValid();
                this.payload = this.bufferPool.acquire(this.payloadLength, false);
                BufferUtil.clearToFill(this.payload);
            }
            ByteBuffer window = buffer.slice();
            int bytesExpected = this.payloadLength - this.payload.position();
            int bytesAvailable = buffer.remaining();
            int windowBytes = Math.min(bytesAvailable, bytesExpected);
            window.limit(window.position() + windowBytes);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Window: {}", BufferUtil.toDetailString(window));
            }
            this.maskProcessor.process(window);
            this.strictnessProcessor.process(window);
            int len = BufferUtil.put(window, this.payload);
            buffer.position(buffer.position() + len);
            if (this.payload.position() < this.payloadLength) continue;
            BufferUtil.flipToFlush(this.payload, 0);
            this.frame.setPayload(this.payload);
            this.payload = null;
            return true;
        }
        return false;
    }

    public void setIncomingFramesHandler(IncomingFrames incoming) {
        this.incomingFramesHandler = incoming;
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Parser[");
        if (this.incomingFramesHandler == null) {
            builder.append("NO_HANDLER");
        } else {
            builder.append(this.incomingFramesHandler.getClass().getSimpleName());
        }
        builder.append(",s=");
        builder.append((Object)this.state);
        builder.append(",c=");
        builder.append(this.cursor);
        builder.append(",len=");
        builder.append(this.payloadLength);
        builder.append(",f=");
        builder.append(this.frame);
        builder.append("]");
        return builder.toString();
    }

    private static enum State {
        START,
        FINOP,
        PAYLOAD_LEN,
        PAYLOAD_LEN_BYTES,
        MASK,
        MASK_BYTES,
        PAYLOAD;

    }
}

