package org.rwtodd.args;

import java.io.PrintStream;
import java.util.Collection;
import java.util.Map;

/**
 * Interface for command-line parameter specs.  The two subclasses
 * are for parameters that are expected on the command-line.  One
 * can also define {@code Param} instances that merely group and
 * format other parameters as part of help text.
 * 
 * @author rwtodd
 */
public sealed interface Param permits OneArgParam, NoArgParam {
    /** Add the parameter's names to a {@code Map<String,Param>}.
     *
     * @param map the {@code Map} to which our names should be added.
     */
    void addToMap(Map<String,Param> map);

    
    /** adds help for this parameter to the given stream.
     * 
     * @param ps the stream to use
     */
    void addHelp(PrintStream ps);

    /**
     * A helper function to format a list of parameter names. It adds
     * the appropriate number of dashes before the names, and separates them.
     *
     * @param names the collection of parameter names.
     * @return a formatted string
     */
    static String formatNames(Collection<String> names) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for(String name: names) {
            if(!first) sb.append(" | ");
            if(name.length() > 1) sb.append('-');
            sb.append('-'); sb.append(name);
            first = false;
        }
        return sb.toString();
    }

    /**
     * A helper function to format a typical parameter.  It uses one line for the names, and
     * an indented line for the description. Descriptions longer than 72 characters are split
     * across lines.  At present, the splitting code doesn't try to cut the lines off at word
     * boundaries... just straight 72-length lines are used.  The algorithm may improve in the
     * future.
     *
     * @param ps the {@code PrintStream} to write to.
     * @param nameLine the line of code for the names, perhaps generated by {@link #formatNames(Collection)}.
     * @param descline the description text for the parameter.
     */
    static void formatTypicalHelp(PrintStream ps, String nameLine, String descline) {
        ps.println(nameLine);
        if(descline.length() <= 72) {
            ps.print("   ");
            ps.println(descline);
        } else {
            int start = 0, end = 72;
            while (start < descline.length()) {
                ps.print("   ");
                ps.println(descline.substring(start, end));
                start = end;
                end = Math.min(end+72, descline.length());
            }
        }
    }
}
