package org.rwtodd.paldesign;

import java.util.Optional;
import java.util.concurrent.ExecutionException;
import javafx.beans.property.ObjectProperty;
import javafx.concurrent.Task;
import javafx.event.ActionEvent;
import javafx.scene.control.ComboBox;
import javafx.fxml.*;
import javafx.scene.control.Button;
import javafx.scene.control.TextField;
import javafx.scene.layout.*;
import javafx.scene.paint.Color;
import javafx.scene.shape.Rectangle;
import javafx.scene.control.ColorPicker;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.MenuItem;
import javafx.scene.image.Image;

/**
 * PaletteDesigner is a custom control to design indexed color palettes.
 *
 * @author richard todd
 */
public class PaletteDesigner extends VBox {
    @FXML private ComboBox<ColorAdder> vintageList;
    @FXML private ColorPicker scaleFrom, scaleTo;
    @FXML private TextField scaleLvl;
    @FXML private Pane colorDisplay;
    @FXML private Pane quantControls;
    @FXML private TextField quantLvl;
    @FXML private Button quantBtn;
    
    private Palette palette;
    
    // the context menu for colors...
    private ContextMenu colormenu;
    private Rectangle selectedColor; // remember who spawned the context menu...
    
    // the image for quantization...
    private ObjectProperty<Image> quantImg;
    
    /**
     * Initializes the control.
     */
    public PaletteDesigner() {
        FXMLLoader loader = new FXMLLoader(getClass().getResource("PaletteDesigner.fxml"));
        loader.setRoot(this);
        loader.setController(this);
        palette = new Palette();
       
        // set up the context menu we'll use...
        colormenu = new ContextMenu();
        MenuItem delMI = new MenuItem("Delete");
        delMI.setOnAction((ae) -> {  
            final Rectangle box = selectedColor;
            final int which = colorDisplay.getChildren().indexOf(box);
            if(which >= 0) {
                palette.remove(which);
                colorDisplay.getChildren().remove(which);
            }
        });
        colormenu.getItems().add(delMI);
        
        // now load our FXML...
        try {
            loader.load();
        } catch(java.io.IOException ex) {
            throw new RuntimeException(ex);
        }
       
        for (VintagePalette vp : VintagePalette.values()) {
            vintageList.getItems().add(new ColorAdder(vp.toString(), vp::getColors));
        }

        // hide the quant controls until the user gives us an image...
        quantControls.setVisible(false);
        quantControls.setManaged(false);
        quantImg = null;
    }        
    
    public void setQuantizationImage(final ObjectProperty<Image> im) {
        quantControls.setVisible(im.get() != null);
        quantControls.setManaged(im.get() != null);
        quantImg = im;
    }
    
    /**
     * Set the palette name and color set. Any existing colors will be cleared.
     * @param name  The optional name of the color set.
     * @param colors The colors to be used as a palette.
     */
    public void setPalette(Optional<String> name, Color[] colors) {
        palette.clear();
        palette.addPalette(name, colors);
        redrawColors();
    }
    
    /**
     * Get the name of the palette generated by the control.
     * @return the palette name.
     */
    public String getPaletteName() {
        return palette.toString();
    }
    
    /**
     * Get the colors generated by the tool.  The colors will be
     * made unique, and at most 256 will be returned.
     * @return The palette, or a monochrome palette if the palette is empty.
     */
    public Color[] getPaletteColors() {
        Color[] answer = palette.toUniqueArray();
        if(answer.length == 0) {
            answer = new Color[] { Color.BLACK, Color.WHITE };
        }
         return answer;  
    }

    @FXML private void clearPalette(ActionEvent ae) {
        palette.clear();
        redrawColors();
    }
    
    @FXML private void dedupPalette(ActionEvent ae) {
        palette.removeDuplicates();
        redrawColors();
    }
    
    @FXML private void addVintage(ActionEvent ae) {
        final ColorAdder ca = vintageList.getValue();
        palette.addPalette(Optional.of(ca.toString()), ca.get());
        redrawColors();
    }
    
    @FXML private void addScale(ActionEvent ae) {
        final int levels = Integer.valueOf(scaleLvl.getText());
        if(levels <= 1) return;
        
        final Color from = scaleFrom.getValue();
        final Color to = scaleTo.getValue();
        final Color[] cols = new Color[levels];
        
        for (int i = 0; i < levels; i++) {
            cols[i] = from.interpolate(to, ((double)i)/(levels-1));
        }
        palette.addPalette(Optional.of("Color Scale"), cols);
        redrawColors();
    }
    
    @FXML private void addQuant(ActionEvent ae) {
        int levels = Integer.valueOf(quantLvl.getText());
        if(levels == 0) return;
        
        quantBtn.setText("Calculating...");
        quantBtn.setDisable(true);
        
        final Task<Color[]> quantTask = new Task<Color[]>() {
            @Override
            protected Color[] call() throws Exception {
               final KMeansQuantizer q = new KMeansQuantizer(quantImg.get(), levels);
               return q.quantize();  
            }
            
        };
        quantTask.setOnSucceeded(t -> {
            try {
               palette.addPalette(Optional.of("Quantized Palette"), quantTask.get());
               quantBtn.setText("Add");
               quantBtn.setDisable(false);
               redrawColors();
            } catch(java.lang.InterruptedException | ExecutionException ie) {
               /* do nothing */
            }
        });
        new Thread(quantTask).start();
    }
    
    private void redrawColors() {
        colorDisplay.getChildren().clear();
        palette.stream()
               .forEachOrdered(c -> {
                    final Rectangle box = new Rectangle(20,20,c);
                    box.setOnMouseClicked((me) -> {
                       selectedColor = (Rectangle)me.getSource();
                       colormenu.show(selectedColor, me.getScreenX(), me.getScreenY());
                    });
                    colorDisplay.getChildren().add(box);
               });
        
                              
    }

    
}
