package org.sackfix.common.objectpool

import scala.collection.mutable

/**
  * Created by Jonathan during 2017.
  *
  * One of the easy performance issues to solve is the number of immutable tuples
  * that get created.   Many of the fields which exist have a small set of values which
  * can be assigned, so, this pool is wierld, it holds immutables and is dynamically created
  * at runtime.  As it builds up knowledge of the fix version it will discard those maps
  * which have too many values, and retain maps of values which belong to a constrained set.
  */
class KeyValPool(val maxSize:Int = 100000, val valueMapMaxSize:Int = 25) {
  // Could mix in synchronizedMap etc but actually do not want that
  val pool = mutable.Map.empty[Int, mutable.Map[String, Tuple2[Int, String]]]
  val tagTooDynamicLookup = mutable.HashSet.empty[Int]
  var currentSize = 0

  def allocTuple(tag: Int, value: String): Tuple2[Int, String] = {
    if (tagTooDynamicLookup.contains(tag)) (tag, value)
    else allocTupleFromPool(tag, value)
  }

  /**
    * Having confirmed that this tag is not too dynamic, eg the quantity field will
    * possibly have a huge number of values, so there is no point caching them to avoid
    * immutable tuple creation.
    */
  private def allocTupleFromPool(tag: Int, value: String) = {
    pool.get(tag) match {
      case Some(valueTagsMap) =>
        valueTagsMap.get(value) match {
          case Some(keyValueTuple) => keyValueTuple
          case None =>
            addToValuesMap(valueTagsMap, tag, value)
        }
      case None =>
        addKeyValueForFirstTime(tag, value)
    }
  }

  /**
    * Called when the pool has never come accross this tag before.
    */
  private def addKeyValueForFirstTime(tag: Int, value: String) : Tuple2[Int, String] = {
    val ret =  Tuple2(tag, value)
    if (currentSize < maxSize) {
      this.synchronized {
        val valueTagsMap = mutable.Map.empty[String, Tuple2[Int, String]]
        pool(tag) = valueTagsMap
        valueTagsMap(value) = ret
        currentSize += 1
      }
    }
    ret
  }

  /**
    * We have a map of values already for this tag.  Try to reuse the value for the tag, or if
    * it is not there, then either
    * - If the number of values is small, add it to the pool
    * - if the number of values is too high, mark this tag as too dynamic to cache, and remove
    * all the previous values from the pool
    */
  private def addToValuesMap(valueTagsMap: mutable.Map[String, Tuple2[Int,String]], tag:Int, value:String) = {
    val ret =  Tuple2(tag, value)
    if (currentSize < maxSize) {
      this.synchronized {
        if (valueTagsMap.size < valueMapMaxSize) {
          valueTagsMap(value) = ret
          currentSize += 1
        } else {
          tagTooDynamicLookup.add(tag)
          valueTagsMap.clear
          pool.remove(tag)
        }
      }
    }
    ret
  }

}