package org.sakaiproject.delegatedaccess.dao.impl;

import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.quartz.JobExecutionException;
import org.sakaiproject.authz.api.AuthzGroupService;
import org.sakaiproject.authz.api.SecurityAdvisor;
import org.sakaiproject.authz.api.SecurityService;
import org.sakaiproject.delegatedaccess.jobs.DelegatedAccessSiteHierarchyJob;
import org.sakaiproject.entity.api.ResourcePropertiesEdit;
import org.sakaiproject.event.api.EventTrackingService;
import org.sakaiproject.event.api.UsageSessionService;
import org.sakaiproject.exception.IdInvalidException;
import org.sakaiproject.exception.IdUsedException;
import org.sakaiproject.exception.PermissionException;
import org.sakaiproject.site.api.Site;
import org.sakaiproject.site.api.SitePage;
import org.sakaiproject.site.api.SiteService;
import org.sakaiproject.tool.api.Session;
import org.sakaiproject.tool.api.SessionManager;
import org.sakaiproject.user.api.User;
import org.sakaiproject.user.api.UserAlreadyDefinedException;
import org.sakaiproject.user.api.UserDirectoryService;
import org.sakaiproject.user.api.UserIdInvalidException;
import org.sakaiproject.user.api.UserNotDefinedException;
import org.sakaiproject.user.api.UserPermissionException;


public class DelegatedAccessSampleDataLoader {
	private SiteService siteService;
	private DelegatedAccessSiteHierarchyJob delegatedAccessSiteHierarchyJob;
	private SecurityService securityService;
	private AuthzGroupService authzGroupService;
	private EventTrackingService eventTrackingService;
	private UsageSessionService usageSessionService;
	private SessionManager sessionManager;
	private UserDirectoryService userDirectoryService;
	private static final Logger log = Logger.getLogger(DelegatedAccessSampleDataLoader.class);

	private List<String> schools = Arrays.asList("MUSIC", "MEDICINE", "EDUCATION");
	private List<String> depts = Arrays.asList("DEPT1", "DEPT2", "DEPT3");
	private List<String> subjs = Arrays.asList("SUBJ1", "SUBJ2","SUBJ3");
	
	public void init(){
		if(siteService == null || securityService == null || delegatedAccessSiteHierarchyJob == null){
			return;
		}
		// Cheating to become admin in order to add sites
		SecurityAdvisor yesMan = new SecurityAdvisor() {
			public SecurityAdvice isAllowed(String userId, String function, String reference) {
				return SecurityAdvice.ALLOWED;
			}
		};
		try{
			loginToSakai();
			securityService.pushAdvisor(yesMan);
			for(String school : schools){
				for(String dept : depts){
					for(String subject : subjs){
						for(int courseNum = 101; courseNum < 600; courseNum += 25){
							String siteid = "DAC-" + school + "-" + dept + "-" + subject + "-" + courseNum; 
							String title = siteid;
							String description = siteid;
							String shortdesc = siteid;

							Site siteEdit = null;
							try {
								siteEdit = siteService.addSite(siteid, "project");
								siteEdit.setTitle(title);
								siteEdit.setDescription(description);
								siteEdit.setShortDescription(shortdesc);
								siteEdit.setPublished(true);
								siteEdit.setType("course");
								
								//for some reason the course template may not work
								if(siteEdit.getTool("sakai.siteinfo") == null){
									SitePage page = siteEdit.addPage();
									page.setTitle("Site Info");
									page.addTool("sakai.siteinfo");
								}
								
								ResourcePropertiesEdit propEdit = siteEdit.getPropertiesEdit();
								propEdit.addProperty("School", school);
								propEdit.addProperty("Department", dept);
								propEdit.addProperty("Subject", subject);
								siteService.save(siteEdit);

							} catch (IdInvalidException e) {
								log.warn(e);
							} catch (IdUsedException e) {
								log.warn(e);
								//this means that we have already ran this, lets quit
								return;
							} catch (PermissionException e) {
								log.warn(e);
							} catch (Exception e){
								//who knows what happened... let's quit!
								log.warn(e);
								return;
							}
						}
					}
				}
			}

			//now that the sites have been added, lets run the hierarhcy job
			try {
				delegatedAccessSiteHierarchyJob.execute(null);
			} catch (JobExecutionException e) {
				log.warn(e);
			}
		
		}catch(Exception e){
			log.warn(e);
		}finally{
			securityService.popAdvisor(yesMan);
			logoutFromSakai();
		}
	}
	
	private void loginToSakai() {
		User user = null;
		try {
			user = userDirectoryService.getUserByEid("datest");
		} catch (UserNotDefinedException e) {
			//user doesn't exist, lets make it:
			try {
				//String id, String eid, String firstName, String lastName, String email, String pw, String type, ResourceProperties properties
				user = userDirectoryService.addUser("datest", "datest", "DA", "Test", "", "datest", "", null);
			} catch (UserIdInvalidException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (UserAlreadyDefinedException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (UserPermissionException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
		}
		if(user != null){
			Session sakaiSession = sessionManager.getCurrentSession();
			sakaiSession.setUserId(user.getId());
			sakaiSession.setUserEid(user.getEid());

			// establish the user's session
			usageSessionService.startSession(user.getId(), "127.0.0.1", "DAtest");

			// update the user's externally provided realm definitions
			authzGroupService.refreshUser(user.getId());

			// post the login event
			eventTrackingService.post(eventTrackingService.newEvent(UsageSessionService.EVENT_LOGIN, null, true));
		}
	}

	private void logoutFromSakai() {
	    Session sakaiSession = sessionManager.getCurrentSession();
		sakaiSession.invalidate();

		// post the logout event
		eventTrackingService.post(eventTrackingService.newEvent(UsageSessionService.EVENT_LOGOUT, null, true));
	}
	
	public void setSiteService(SiteService siteService) {
		this.siteService = siteService;
	}

	public SiteService getSiteService() {
		return siteService;
	}

	public DelegatedAccessSiteHierarchyJob getDelegatedAccessSiteHierarchyJob() {
		return delegatedAccessSiteHierarchyJob;
	}

	public void setDelegatedAccessSiteHierarchyJob(
			DelegatedAccessSiteHierarchyJob delegatedAccessSiteHierarchyJob) {
		this.delegatedAccessSiteHierarchyJob = delegatedAccessSiteHierarchyJob;
	}

	public SecurityService getSecurityService() {
		return securityService;
	}

	public void setSecurityService(SecurityService securityService) {
		this.securityService = securityService;
	}

	public AuthzGroupService getAuthzGroupService() {
		return authzGroupService;
	}

	public void setAuthzGroupService(AuthzGroupService authzGroupService) {
		this.authzGroupService = authzGroupService;
	}

	public EventTrackingService getEventTrackingService() {
		return eventTrackingService;
	}

	public void setEventTrackingService(EventTrackingService eventTrackingService) {
		this.eventTrackingService = eventTrackingService;
	}

	public UsageSessionService getUsageSessionService() {
		return usageSessionService;
	}

	public void setUsageSessionService(UsageSessionService usageSessionService) {
		this.usageSessionService = usageSessionService;
	}

	public SessionManager getSessionManager() {
		return sessionManager;
	}

	public void setSessionManager(SessionManager sessionManager) {
		this.sessionManager = sessionManager;
	}

	public UserDirectoryService getUserDirectoryService() {
		return userDirectoryService;
	}

	public void setUserDirectoryService(UserDirectoryService userDirectoryService) {
		this.userDirectoryService = userDirectoryService;
	}
}
