/*
 * ----------------------------------------------------------------------------
 * This file is part of JSBML. Please visit <http://sbml.org/Software/JSBML>
 * for the latest version of JSBML and more information about SBML.
 * 
 * Copyright (C) 2009-2022 jointly by the following organizations:
 * 1. The University of Tuebingen, Germany
 * 2. EMBL European Bioinformatics Institute (EBML-EBI), Hinxton, UK
 * 3. The California Institute of Technology, Pasadena, CA, USA
 * 4. The University of California, San Diego, La Jolla, CA, USA
 * 5. The Babraham Institute, Cambridge, UK
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation. A copy of the license agreement is provided
 * in the file named "LICENSE.txt" included with this software distribution
 * and also available online as <http://sbml.org/Software/JSBML/License>.
 * ----------------------------------------------------------------------------
 */
package org.sbml.jsbml.validator.offline.constraints;

import java.util.Set;

import org.sbml.jsbml.ext.layout.CubicBezier;
import org.sbml.jsbml.ext.layout.LayoutConstants;
import org.sbml.jsbml.validator.SBMLValidator.CHECK_CATEGORY;
import org.sbml.jsbml.validator.offline.ValidationContext;
import org.sbml.jsbml.validator.offline.constraints.helper.DuplicatedElementValidationFunction;
import org.sbml.jsbml.validator.offline.constraints.helper.UnknownCoreAttributeValidationFunction;
import org.sbml.jsbml.validator.offline.constraints.helper.UnknownCoreElementValidationFunction;
import org.sbml.jsbml.validator.offline.constraints.helper.UnknownPackageAttributeValidationFunction;
import org.sbml.jsbml.validator.offline.constraints.helper.UnknownPackageElementValidationFunction;;

/**
 * Defines validation rules (as {@link ValidationFunction} instances) for the {@link CubicBezier} class.
 *  
 * @author rodrigue
 * @since 1.3
 */
public class CubicBezierConstraints extends AbstractConstraintDeclaration {

  /* (non-Javadoc)
   * @see org.sbml.jsbml.validator.offline.constraints.ConstraintDeclaration#addErrorCodesForAttribute(java.util.Set, int, int, java.lang.String)
   */
  @Override
  public void addErrorCodesForAttribute(Set<Integer> set, int level,
    int version, String attributeName, ValidationContext context) 
  {
    // no specific attribute, so nothing to do.

  }

  /* (non-Javadoc)
   * @see org.sbml.jsbml.validator.offline.constraints.ConstraintDeclaration#addErrorCodesForCheck(java.util.Set, int, int, org.sbml.jsbml.validator.SBMLValidator.CHECK_CATEGORY)
   */
  @Override
  public void addErrorCodesForCheck(Set<Integer> set, int level, int version,
    CHECK_CATEGORY category, ValidationContext context) {

    switch (category) {
    case GENERAL_CONSISTENCY:

      addRangeToSet(set, LAYOUT_21601, LAYOUT_21604 );
      
      break;
    case IDENTIFIER_CONSISTENCY:
      break;
    case MATHML_CONSISTENCY:
      break;
    case MODELING_PRACTICE:
      break;
    case OVERDETERMINED_MODEL:
      break;
    case SBO_CONSISTENCY:
      break;
    case UNITS_CONSISTENCY:
      break;
    }
  }


  @Override
  public ValidationFunction<?> getValidationFunction(int errorCode, ValidationContext context) {
    ValidationFunction<CubicBezier> func = null;

    switch (errorCode) {

    case LAYOUT_21601:
    {
      func = new UnknownCoreElementValidationFunction<CubicBezier>();
      break;
    }
    case LAYOUT_21602:
    {
      func = new UnknownCoreAttributeValidationFunction<CubicBezier>();
      break;
    }
    case LAYOUT_21603:
    {
      func = new ValidationFunction<CubicBezier>() {

        @Override
        public boolean check(ValidationContext ctx, CubicBezier cubicBezier) {
          
          return cubicBezier.isSetStart() && cubicBezier.isSetEnd() && cubicBezier.isSetBasePoint1() && cubicBezier.isSetBasePoint2()
              && new DuplicatedElementValidationFunction<CubicBezier>(LayoutConstants.start).check(ctx, cubicBezier)
              && new DuplicatedElementValidationFunction<CubicBezier>(LayoutConstants.end).check(ctx, cubicBezier)
              && new DuplicatedElementValidationFunction<CubicBezier>(LayoutConstants.basePoint1).check(ctx, cubicBezier)
              && new DuplicatedElementValidationFunction<CubicBezier>(LayoutConstants.basePoint2).check(ctx, cubicBezier)
              && new UnknownPackageElementValidationFunction<CubicBezier>(LayoutConstants.shortLabel).check(ctx, cubicBezier);
        }
      };
      break;
    }
    case LAYOUT_21604:
    {
      func = new UnknownPackageAttributeValidationFunction<CubicBezier>(LayoutConstants.shortLabel);
      break;
    }
    }

    return func;
  }
}
