package metadoc.server

import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths

/**
  * A sourcepath contains `*.{class,java,scala}` files of a project.
  *
  * @param classpath the regular JVM classpath of this project, containing
  *                  `*.class` files that can be used to compile the project.
  *                  Must match the `-classpath` argument passed to the Scala
  *                  compiler.
  * @param sources the accompanying sources.jar files for this project that
  *                are published under the "sources" classifier.
  */
case class Sourcepath(classpath: List[Path], sources: List[Path])

object Sourcepath {

  /** Use coursier to fetch the classpath and sources of an artifact.
    *
    * @param artifact the artifact name, for example
    *                 - org.scala-lang:scala-library:2.12.7
    *                 - org.scalameta:scalameta_2.12:4.0.0
    */
  def apply(artifact: String): Sourcepath = {
    Sourcepath(
      classpath = coursierFetch(artifact, Nil),
      sources = jdkSources().toList ++
        coursierFetch(artifact, List("--classifier", "sources"))
    )
  }

  /** The sources of the JDK, for example `java/lang/String.java` */
  def jdkSources(): Option[Path] = {
    for {
      javaHome <- sys.props.get("java.home")
      srcZip = Paths.get(javaHome).getParent.resolve("src.zip")
      if Files.isRegularFile(srcZip)
    } yield srcZip
  }

  private def coursierFetch(
      artifact: String,
      extra: List[String]
  ): List[Path] = {
    sys.process
      .Process(List("coursier", "fetch", artifact) ++ extra)
      .!!
      .trim
      .linesIterator
      .map(jar => Paths.get(jar))
      .toList
  }
}
