package scala.meta.internal.metals

import java.nio.file.Path
import org.eclipse.lsp4j.SymbolKind
import org.eclipse.lsp4j.jsonrpc.CancelChecker
import org.eclipse.{lsp4j => l}
import scala.collection.mutable
import scala.meta.internal.metals.MetalsEnrichments._
import scala.meta.internal.mtags.OnDemandSymbolIndex
import scala.meta.internal.mtags.Symbol
import scala.meta.internal.mtags.SymbolDefinition
import scala.meta.internal.semanticdb.Scala.Descriptor
import scala.meta.internal.semanticdb.Scala.DescriptorParser
import scala.meta.internal.semanticdb.Scala.Symbols
import scala.meta.io.AbsolutePath
import scala.meta.pc.SymbolSearchVisitor
import org.eclipse.lsp4j.SymbolKind
import java.{util => ju}

/**
 * A symbol search visitor for `workspace/symbol`.
 *
 * - workspace symbols are converted directly to l.SymbolInformation
 * - classpath symbols are converted into "goto definition" requests,
 *   which creates files on disk, and then into l.SymbolInformation.
 */
class WorkspaceSearchVisitor(
    workspace: AbsolutePath,
    query: WorkspaceSymbolQuery,
    token: CancelChecker,
    index: OnDemandSymbolIndex,
    fileOnDisk: AbsolutePath => AbsolutePath
) extends SymbolSearchVisitor {
  private val fromWorkspace = new ju.ArrayList[l.SymbolInformation]()
  private val fromClasspath = new ju.ArrayList[l.SymbolInformation]()
  private val bufferedClasspath = new ju.ArrayList[(String, String)]()
  def allResults(): Seq[l.SymbolInformation] = {
    if (fromWorkspace.isEmpty) {
      bufferedClasspath.forEach {
        case (pkg, name) =>
          expandClassfile(pkg, name)
      }
    }

    fromWorkspace.sort(byNameLength)
    fromClasspath.sort(byNameLength)

    val result = new ju.ArrayList[l.SymbolInformation]()
    result.addAll(fromWorkspace)
    result.addAll(fromClasspath)

    if (!bufferedClasspath.isEmpty && fromClasspath.isEmpty) {
      val dependencies = workspace.resolve(Directories.workspaceSymbol)
      if (!dependencies.isFile) {
        dependencies.writeText(Messages.WorkspaceSymbolDependencies.title)
      }
      result.add(
        new l.SymbolInformation(
          Messages.WorkspaceSymbolDependencies.title,
          // NOTE(olafur) The "Event" symbol kind is arbitrarily picked, in VS
          // Code its icon is a yellow lightning which makes it similar but
          // distinct enough from the regular results. I tried the "File" kind
          // but found the icon in VS Code to be ugly and its white color
          // attracted too much attention.
          SymbolKind.Event,
          new l.Location(
            dependencies.toURI.toString(),
            new l.Range(new l.Position(0, 0), new l.Position(0, 0))
          )
        )
      )
    }
    result.asScala
  }
  private val byNameLength = new ju.Comparator[l.SymbolInformation] {
    def compare(x: l.SymbolInformation, y: l.SymbolInformation): Int = {
      Integer.compare(x.getName().length(), y.getName().length())
    }
  }
  val isVisited = mutable.Set.empty[AbsolutePath]
  def definition(
      pkg: String,
      filename: String,
      index: OnDemandSymbolIndex
  ): Option[SymbolDefinition] = {
    val nme = Classfile.name(filename)
    val tpe = Symbol(Symbols.Global(pkg, Descriptor.Type(nme)))
    index.definition(tpe).orElse {
      val term = Symbol(Symbols.Global(pkg, Descriptor.Term(nme)))
      index.definition(term)
    }
  }
  override def shouldVisitPackage(pkg: String): Boolean = true
  override def visitWorkspaceSymbol(
      path: Path,
      symbol: String,
      kind: SymbolKind,
      range: l.Range
  ): Int = {
    val (desc, owner) = DescriptorParser(symbol)
    fromWorkspace.add(
      new l.SymbolInformation(
        desc.name.value,
        kind,
        new l.Location(path.toUri.toString, range),
        owner.replace('/', '.')
      )
    )
    1
  }
  override def visitClassfile(pkg: String, filename: String): Int = {
    if (fromWorkspace.isEmpty || query.isClasspath) {
      expandClassfile(pkg, filename)
    } else {
      bufferedClasspath.add(pkg -> filename)
      1
    }
  }
  override def isCancelled: Boolean = token.isCancelled
  private def expandClassfile(pkg: String, filename: String): Int = {
    var isHit = false
    for {
      defn <- definition(pkg, filename, index)
      if !isVisited(defn.path)
    } {
      isVisited += defn.path
      val input = defn.path.toInput
      lazy val uri = fileOnDisk(defn.path).toURI.toString
      SemanticdbDefinition.foreach(input) { defn =>
        if (query.matches(defn.info)) {
          fromClasspath.add(defn.toLSP(uri))
          isHit = true
        }
      }
    }
    if (isHit) 1 else 0
  }
}
