package molecule.datomic.base.marshalling

import java.io.StringReader
import java.net.URI
import java.util
import java.util.{Collections, Date, UUID, List => jList}
import datomic.Peer.toT
import datomic.Util._
import datomic.{Util, Database => PeerDb, Datom => PeerDatom}
import datomicClient.ClojureBridge
import datomicScala.client.api.{Datom => ClientDatom}
import molecule.core.exceptions.MoleculeException
import molecule.core.marshalling._
import molecule.core.marshalling.nodes.Obj
import molecule.core.util.testing.TimerPrint
import molecule.core.util.{DateHandling, Helpers, JavaConversions, Quoted}
import molecule.datomic.base.api.DatomicEntity
import molecule.datomic.base.facade._
import molecule.datomic.base.marshalling.packers.PackEntityGraph
import molecule.datomic.client.facade.{Conn_Client, DatomicDb_Client, Datomic_DevLocal, Datomic_PeerServer}
import molecule.datomic.peer.facade.{Conn_Peer, DatomicDb_Peer, Datomic_Peer}
import scala.collection.mutable
import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal

case class DatomicRpc()(implicit ec: ExecutionContext) extends MoleculeRpc
  with DateHandling with DateStrLocal
  with Helpers with ClojureBridge
  with PackEntityGraph with Quoted
  with BooPicklers
  with PackBase
  with JavaConversions {

  // Necessary for `readString` to encode uri in transactions
  require("clojure.core.async")

  // Api ---------------------------------------------

  def transact(
    connProxy: ConnProxy,
    stmtsEdn: String,
    uriAttrs: Set[String]
  ): Future[TxReportRPC] = {
    for {
      conn <- getConn(connProxy)

      _ = println(stmtsEdn)

      javaStmts = getJavaStmts(stmtsEdn, uriAttrs)
      txReport <- conn.transact(javaStmts)
    } yield {
      TxReportRPC(txReport.t, txReport.tx, txReport.txInstant, txReport.eids, txReport.txData, txReport.toString)
    }
  }

  def query2packed(
    connProxy: ConnProxy,
    datalogQuery: String,
    rules: Seq[String],
    l: Seq[(Int, String, String)],
    ll: Seq[(Int, String, Seq[String])],
    lll: Seq[(Int, String, Seq[Seq[String]])],
    maxRows0: Int,
    obj: Obj,
    nestedLevels: Int,
    isOptNested: Boolean,
    refIndexes: List[List[Int]],
    tacitIndexes: List[List[Int]]
  ): Future[String] = Future {
    try {
      val log = new log
      val t   = TimerPrint("DatomicRpc")

      //    println("------------------------------")
      //    println("================================================================================")
      //      println(datalogQuery)
      //      if (rules.nonEmpty) {
      //        println("Rules:")
      //        rules foreach println
      //      }
      //      println("l  : " + l)
      //      println("ll : " + ll)
      //      println("lll: " + lll)

      val inputs    = unmarshallInputs(l ++ ll ++ lll)
      val allInputs = if (rules.nonEmpty) rules ++ inputs else inputs

      //      inputs.foreach(i => println(s"$i   " + i.getClass))

      //      if (inputs.nonEmpty) {
      //        println("----------")
      //        println(inputs.head.asInstanceOf[jList[_]].size)
      //        println(inputs.head.asInstanceOf[jList[_]].get(0))
      //      }

      for {
        conn <- getConn(connProxy)
        allRows <- conn.rawQuery(datalogQuery, allInputs)
      } yield {
        val rowCountAll = allRows.size
        val maxRows     = if (maxRows0 == -1 || rowCountAll < maxRows0) rowCountAll else maxRows0
        val queryTime   = t.delta
        val space       = " " * (70 - datalogQuery.split('\n').last.length)
        val time        = qTime(queryTime)
        val timeRight   = " " * (8 - time.length) + time

        log("================================================================================")
        log(datalogQuery + space + timeRight)
        if (allInputs.nonEmpty)
          log(allInputs.mkString("Inputs:\n", "\n", ""))

        //      log(datalogQuery + space + timeRight + "  " + conn.asInstanceOf[Conn_Peer].peerConn.db)
        //      log(s"\n---- Querying Datomic... --------------------")
        //      log(datalogQuery)
        //      log(qTime(queryTime) + "  " + datalogQuery)
        //      log("connProxy uuid: " + connProxy.uuid)
        //      log("Query time  : " + thousands(queryTime) + " ms")
        //      log("rowCountAll : " + rowCountAll)
        //      log("maxRows     : " + (if (maxRows == -1) "all" else maxRows))
        //      log("rowCount    : " + rowCount)

        log("-------------------------------")
        //      log(obj.toString)
        //      log("-------------------------------")
        //      log(refIndexes.mkString("\n"))
        //      log("-------------------------------")
        //      log(tacitIndexes.mkString("\n"))
        //      log("-------------------------------")
        allRows.forEach(row => log(row.toString))
        log.print

        val packed = if (isOptNested) {
          OptNested2packed(obj, allRows, maxRows, refIndexes, tacitIndexes).getPacked
        } else if (nestedLevels == 0) {
          // Flat and composites
          Flat2packed(obj, allRows, maxRows).getPacked
        } else {
          Nested2packed(obj, allRows, nestedLevels).getPacked
        }

        //        println("-------------------------------" + packed)
        //        log("Sending data to client... Total server time: " + t.msTotal)
        packed
      }
    } catch {
      case NonFatal(exc) => Future.failed(exc)
    }
  }.flatten

  // Unmarshall to Datomic java types
  private def unmarshallInputs(lists: Seq[(Int, String, Any)]): Seq[Object] = {
    lists.sortBy(_._1).map {
      case (_, tpe, rawValue) =>
        val cast = tpe match {
          case "String"     => if (isEnum(rawValue)) (v: String) => getEnum(v) else (v: String) => v
          case "Int"        => (v: String) => new java.lang.Long(v)
          case "Long"       => (v: String) => new java.lang.Long(v)
          case "Double"     => (v: String) => new java.lang.Double(v)
          case "Boolean"    => (v: String) => v.toBoolean.asInstanceOf[Object]
          case "Date"       => (v: String) => str2date(v).asInstanceOf[Object]
          case "URI"        => (v: String) => new java.net.URI(v).asInstanceOf[Object]
          case "UUID"       => (v: String) => java.util.UUID.fromString(v).asInstanceOf[Object]
          case "BigInt"     => (v: String) => new java.math.BigInteger(v).asInstanceOf[Object]
          case "BigDecimal" => (v: String) => new java.math.BigDecimal(v).asInstanceOf[Object]
          case "Any"        => (s: String) =>
            val v = s.drop(10)
            s.take(10) match {
              case "String    " => if (isEnum(v)) getEnum(v) else v
              case "Int       " => new java.lang.Long(v)
              case "Long      " => new java.lang.Long(v)
              case "Double    " => new java.lang.Double(v)
              case "Boolean   " => v.toBoolean.asInstanceOf[Object]
              case "Date      " => str2date(v).asInstanceOf[Object]
              case "URI       " => new URI(v).asInstanceOf[Object]
              case "UUID      " => UUID.fromString(v).asInstanceOf[Object]
              case "BigInt    " => new java.math.BigInteger(v).asInstanceOf[Object]
              case "BigDecimal" => new java.math.BigDecimal(v).asInstanceOf[Object]
            }
          case _            => throw MoleculeException(s"Unexpected type to cast: $tpe")
        }

        rawValue match {
          case l: Seq[_] =>
            Util.list(l.collect {
              case l2: Seq[_] =>
                val Seq(k, v2: String) = l2
                Util.list(k.toString.asInstanceOf[Object], cast(v2))

              case v1: String => cast(v1)
            }: _*)

          case v: String => cast(v)
          case _         => throw MoleculeException("Unexpected input values")
        }
    }
  }

  private def isEnum(rawValue: Any) = {
    // Check enum prefix on any of 3 possible levels
    rawValue match {
      case l: Seq[_] => l.headOption.fold(false) {
        case l2: Seq[_] => l2.headOption.fold(false)(v => v.toString.startsWith("__enum__"))
        case v: String  => v.startsWith("__enum__")
      }
      case v: String => v.startsWith("__enum__")
      case _         => false
    }
  }

  private def getEnum(s: String): AnyRef = s match {
    case r"__enum__:([A-Za-z0-9\._]+)$ns/([A-Za-z0-9_]+)$enum" => clojure.lang.Keyword.intern(ns, enum)
    case other                                                 =>
      throw MoleculeException(s"Unexpected enum input: `$other`")
  }

  private def getIdent(conn: Conn, eid: Any): Future[String] = {
    conn.rawQuery(s"[:find ?idIdent :where [$eid :db/ident ?idIdent]]").map { rows =>
      if (rows.size() != 1)
        throw MoleculeException(s"Couldn't find attribute name of eid $eid in saved schema.")
      else
        rows.iterator().next().get(0).toString
    }
  }

  def index2packed(
    connProxy: ConnProxy,
    api: String,
    index: String,
    args: IndexArgs,
    attrs: Seq[String]
  ): Future[String] = {
    def castTpeV(tpe: String, v: String): Object = {
      (tpe, v) match {
        case ("String", v)     => if (isEnum(v)) getEnum(v) else v
        case ("Int", v)        => v.toInt.asInstanceOf[Object]
        case ("Long", v)       => v.toLong.asInstanceOf[Object]
        case ("Double", v)     => v.toDouble.asInstanceOf[Object]
        case ("Boolean", v)    => v.toBoolean.asInstanceOf[Object]
        case ("Date", v)       => str2date(v).asInstanceOf[Object]
        case ("UUID", v)       => java.util.UUID.fromString(v).asInstanceOf[Object]
        case ("URI", v)        => new java.net.URI(v).asInstanceOf[Object]
        case ("BigInt", v)     => new java.math.BigInteger(v).asInstanceOf[Object]
        case ("BigDecimal", v) =>
          val v1 = if (v.contains(".")) v else s"$v.0"
          new java.math.BigDecimal(v1).asInstanceOf[Object]
        case _                 => throw MoleculeException(s"Unexpected input pair to cast: ($tpe, $v)")
      }
    }

    def datomArgs: Seq[Any] = index match {
      case "EAVT" => args match {
        case IndexArgs(-1L, "", "", "", -1L, -1L, _, _) => Nil
        case IndexArgs(e, "", "", "", -1L, -1L, _, _)   => Seq(e)
        case IndexArgs(e, a, "", "", -1L, -1L, _, _)    => Seq(e, read(a))
        case IndexArgs(e, a, v, tpe, -1L, -1L, _, _)    => Seq(e, read(a), castTpeV(tpe, v))
        case IndexArgs(e, a, v, tpe, t, -1L, _, _)      => Seq(e, read(a), castTpeV(tpe, v), t)
        case IndexArgs(e, a, v, tpe, -1L, inst, _, _)   => Seq(e, read(a), castTpeV(tpe, v), new Date(inst))
        case other                                      => throw MoleculeException("Unexpected IndexArgs: " + other)
      }
      case "AEVT" => args match {
        case IndexArgs(-1L, "", "", "", -1L, -1L, _, _) => Nil
        case IndexArgs(-1L, a, "", "", -1L, -1L, _, _)  => Seq(read(a))
        case IndexArgs(e, a, "", "", -1L, -1L, _, _)    => Seq(read(a), e)
        case IndexArgs(e, a, v, tpe, -1L, -1L, _, _)    => Seq(read(a), e, castTpeV(tpe, v))
        case IndexArgs(e, a, v, tpe, t, -1L, _, _)      => Seq(read(a), e, castTpeV(tpe, v), t)
        case IndexArgs(e, a, v, tpe, -1L, inst, _, _)   => Seq(read(a), e, castTpeV(tpe, v), new Date(inst))
        case other                                      => throw MoleculeException("Unexpected IndexArgs: " + other)
      }
      case "AVET" => args match {
        case IndexArgs(-1L, "", "", "", -1L, -1L, _, _) => Nil
        case IndexArgs(-1L, a, "", "", -1L, -1L, _, _)  => Seq(read(a))
        case IndexArgs(-1L, a, v, tpe, -1L, -1L, _, _)  => Seq(read(a), castTpeV(tpe, v))
        case IndexArgs(e, a, v, tpe, -1L, -1L, _, _)    => Seq(read(a), castTpeV(tpe, v), e)
        case IndexArgs(e, a, v, tpe, t, -1L, _, _)      => Seq(read(a), castTpeV(tpe, v), e, t)
        case IndexArgs(e, a, v, tpe, -1L, inst, _, _)   => Seq(read(a), castTpeV(tpe, v), e, new Date(inst))
        case other                                      => throw MoleculeException("Unexpected IndexArgs: " + other)
      }
      case "VAET" => args match {
        case IndexArgs(-1L, "", "", "", -1L, -1L, _, _) => Nil
        case IndexArgs(-1L, "", v, tpe, -1L, -1L, _, _) => Seq(castTpeV(tpe, v))
        case IndexArgs(-1L, a, v, tpe, -1L, -1L, _, _)  => Seq(castTpeV(tpe, v), read(a))
        case IndexArgs(e, a, v, tpe, -1L, -1L, _, _)    => Seq(castTpeV(tpe, v), read(a), e)
        case IndexArgs(e, a, v, tpe, t, -1L, _, _)      => Seq(castTpeV(tpe, v), read(a), e, t)
        case IndexArgs(e, a, v, tpe, -1L, inst, _, _)   => Seq(castTpeV(tpe, v), read(a), e, new Date(inst))
        case other                                      => throw MoleculeException("Unexpected IndexArgs: " + other)
      }
      case other  => throw MoleculeException("Unexpected index name: " + other)
    }

    try {
      for {
        conn <- getConn(connProxy)
        db <- conn.db
        packed <- {
          val adhocDb: DatomicDb = db
          lazy val attrMap = conn.connProxy.attrMap ++ Seq(":db/txInstant" -> (1, "Date"))

          def peerDatomElement2packed(
            tOpt: Option[Long],
            attr: String
          ): (StringBuffer, PeerDatom) => Future[StringBuffer] = attr match {
            case "e"                   => (sb: StringBuffer, d: PeerDatom) => Future(add(sb, d.e.toString))
            case "a"                   => (sb: StringBuffer, d: PeerDatom) =>
              Future {
                add(sb, adhocDb.getDatomicDb.asInstanceOf[PeerDb].ident(d.a).toString)
                end(sb)
              }
            case "v"                   => (sb: StringBuffer, d: PeerDatom) =>
              Future {
                val a         = adhocDb.getDatomicDb.asInstanceOf[PeerDb].ident(d.a).toString
                val (_, tpe)  = attrMap.getOrElse(a,
                  throw MoleculeException(s"Unexpected attribute `$a` not found in attrMap.")
                )
                val tpePrefix = tpe + " " * (10 - tpe.length)
                tpe match {
                  case "String" => add(sb, tpePrefix + d.v.toString); end(sb)
                  case "Date"   => add(sb, tpePrefix + date2str(d.v.asInstanceOf[Date]))
                  case _        => add(sb, tpePrefix + d.v.toString)
                }
              }
            case "t" if tOpt.isDefined => (sb: StringBuffer, _: PeerDatom) => Future(add(sb, tOpt.get.toString))
            case "t"                   => (sb: StringBuffer, d: PeerDatom) => Future(add(sb, toT(d.tx).toString))
            case "tx"                  => (sb: StringBuffer, d: PeerDatom) => Future(add(sb, d.tx.toString))
            case "txInstant"           => (sb: StringBuffer, d: PeerDatom) =>
              adhocDb.entity(conn, d.tx).rawValue(":db/txInstant").map { v =>
                add(sb, date2str(v.asInstanceOf[Date]))
              }
            case "op"                  => (sb: StringBuffer, d: PeerDatom) => Future(add(sb, d.added.toString))
            case x                     => throw MoleculeException("Unexpected PeerDatom element: " + x)
          }


          def clientDatomElement2packed(
            tOpt: Option[Long],
            attr: String
          ): (StringBuffer, ClientDatom) => Future[StringBuffer] = {
            attr match {
              case "e" => (sb: StringBuffer, d: ClientDatom) => Future(add(sb, d.e.toString))
              case "a" => (sb: StringBuffer, d: ClientDatom) =>
                getIdent(conn, d.a).map { attrName =>
                  add(sb, attrName)
                  end(sb)
                }

              case "v" => (sb: StringBuffer, d: ClientDatom) =>
                getIdent(conn, d.a).map { attrName =>
                  val (_, tpe)  = attrMap.getOrElse(attrName,
                    throw MoleculeException(s"Attribute name `$attrName` not found in attrMap.")
                  )
                  val tpePrefix = tpe + " " * (10 - tpe.length)
                  tpe match {
                    case "String" => add(sb, tpePrefix + d.v.toString); end(sb)
                    case "Date"   => add(sb, tpePrefix + date2str(d.v.asInstanceOf[Date]))
                    case _        => add(sb, tpePrefix + d.v.toString)
                  }
                }

              case "t" if tOpt.isDefined => (sb: StringBuffer, _: ClientDatom) => Future(add(sb, tOpt.get.toString))
              case "t"                   => (sb: StringBuffer, d: ClientDatom) => Future(add(sb, toT(d.tx).toString))
              case "tx"                  => (sb: StringBuffer, d: ClientDatom) => Future(add(sb, d.tx.toString))
              case "txInstant"           => (sb: StringBuffer, d: ClientDatom) =>
                adhocDb.entity(conn, d.tx).rawValue(":db/txInstant").map { v =>
                  add(sb, date2str(v.asInstanceOf[Date]))
                }
              case "op"                  => (sb: StringBuffer, d: ClientDatom) => Future(add(sb, d.added.toString))
              case x                     => throw MoleculeException("Unexpected ClientDatom element: " + x)
            }
          }

          def getPeerDatom2packed(tOpt: Option[Long]): (StringBuffer, PeerDatom) => Future[StringBuffer] = attrs.length match {
            case 1 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                } yield sb1

            case 2 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                } yield sb2

            case 3 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              val x3 = peerDatomElement2packed(tOpt, attrs(2))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                } yield sb3

            case 4 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              val x3 = peerDatomElement2packed(tOpt, attrs(2))
              val x4 = peerDatomElement2packed(tOpt, attrs(3))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                } yield sb4

            case 5 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              val x3 = peerDatomElement2packed(tOpt, attrs(2))
              val x4 = peerDatomElement2packed(tOpt, attrs(3))
              val x5 = peerDatomElement2packed(tOpt, attrs(4))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                } yield sb5

            case 6 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              val x3 = peerDatomElement2packed(tOpt, attrs(2))
              val x4 = peerDatomElement2packed(tOpt, attrs(3))
              val x5 = peerDatomElement2packed(tOpt, attrs(4))
              val x6 = peerDatomElement2packed(tOpt, attrs(5))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                  sb6 <- x6(sb5, d)
                } yield sb6

            case 7 =>
              val x1 = peerDatomElement2packed(tOpt, attrs.head)
              val x2 = peerDatomElement2packed(tOpt, attrs(1))
              val x3 = peerDatomElement2packed(tOpt, attrs(2))
              val x4 = peerDatomElement2packed(tOpt, attrs(3))
              val x5 = peerDatomElement2packed(tOpt, attrs(4))
              val x6 = peerDatomElement2packed(tOpt, attrs(5))
              val x7 = peerDatomElement2packed(tOpt, attrs(6))
              (sb: StringBuffer, d: PeerDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                  sb6 <- x6(sb5, d)
                  sb7 <- x7(sb6, d)
                } yield sb7
          }

          def getClientDatom2packed(tOpt: Option[Long]): (StringBuffer, ClientDatom) => Future[StringBuffer] = attrs.length match {
            case 1 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                } yield sb1

            case 2 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                } yield sb2

            case 3 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              val x3 = clientDatomElement2packed(tOpt, attrs(2))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                } yield sb3

            case 4 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              val x3 = clientDatomElement2packed(tOpt, attrs(2))
              val x4 = clientDatomElement2packed(tOpt, attrs(3))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                } yield sb4

            case 5 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              val x3 = clientDatomElement2packed(tOpt, attrs(2))
              val x4 = clientDatomElement2packed(tOpt, attrs(3))
              val x5 = clientDatomElement2packed(tOpt, attrs(4))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                } yield sb5

            case 6 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              val x3 = clientDatomElement2packed(tOpt, attrs(2))
              val x4 = clientDatomElement2packed(tOpt, attrs(3))
              val x5 = clientDatomElement2packed(tOpt, attrs(4))
              val x6 = clientDatomElement2packed(tOpt, attrs(5))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                  sb6 <- x6(sb5, d)
                } yield sb6

            case 7 =>
              val x1 = clientDatomElement2packed(tOpt, attrs.head)
              val x2 = clientDatomElement2packed(tOpt, attrs(1))
              val x3 = clientDatomElement2packed(tOpt, attrs(2))
              val x4 = clientDatomElement2packed(tOpt, attrs(3))
              val x5 = clientDatomElement2packed(tOpt, attrs(4))
              val x6 = clientDatomElement2packed(tOpt, attrs(5))
              val x7 = clientDatomElement2packed(tOpt, attrs(6))
              (sb: StringBuffer, d: ClientDatom) =>
                for {
                  sb1 <- x1(sb, d)
                  sb2 <- x2(sb1, d)
                  sb3 <- x3(sb2, d)
                  sb4 <- x4(sb3, d)
                  sb5 <- x5(sb4, d)
                  sb6 <- x6(sb5, d)
                  sb7 <- x7(sb6, d)
                } yield sb7
          }

          // Pack Datoms
          val sbFut = api match {
            case "datoms" =>
              adhocDb match {
                case adhocDb: DatomicDb_Peer   =>
                  val datomicIndex = index match {
                    case "EAVT" => datomic.Database.EAVT
                    case "AEVT" => datomic.Database.AEVT
                    case "AVET" => datomic.Database.AVET
                    case "VAET" => datomic.Database.VAET
                  }
                  val datom2packed = getPeerDatom2packed(None)
                  adhocDb.datoms(datomicIndex, datomArgs: _*).flatMap { datoms =>
                    datoms.asScala.foldLeft(Future(new StringBuffer())) {
                      case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                    }
                  }
                case adhocDb: DatomicDb_Client =>
                  val datomicIndex = index match {
                    case "EAVT" => ":eavt"
                    case "AEVT" => ":aevt"
                    case "AVET" => ":avet"
                    case "VAET" => ":vaet"
                  }
                  val datom2packed = getClientDatom2packed(None)
                  adhocDb.datoms(datomicIndex, datomArgs).flatMap { datoms =>
                    datoms.iterator().asScala.foldLeft(Future(new StringBuffer())) {
                      case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                    }
                  }
              }

            case "indexRange" =>
              adhocDb match {
                case adhocDb: DatomicDb_Peer   =>
                  val datom2packed = getPeerDatom2packed(None)
                  val startValue   = if (args.v.isEmpty) null else castTpeV(args.tpe, args.v)
                  val endValue     = if (args.v2.isEmpty) null else castTpeV(args.tpe2, args.v2)
                  adhocDb.indexRange(args.a, startValue, endValue).flatMap { datoms =>
                    datoms.asScala.foldLeft(Future(new StringBuffer())) {
                      case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                    }
                  }
                case adhocDb: DatomicDb_Client =>
                  val datom2packed = getClientDatom2packed(None)
                  val startValue   = if (args.v.isEmpty) None else Some(castTpeV(args.tpe, args.v))
                  val endValue     = if (args.v2.isEmpty) None else Some(castTpeV(args.tpe2, args.v2))
                  adhocDb.indexRange(args.a, startValue, endValue).flatMap { datoms =>
                    datoms.iterator().asScala.foldLeft(Future(new StringBuffer())) {
                      case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                    }
                  }
              }

            case "txRange" =>
              // Loop transactions
              conn match {
                case conn: Conn_Peer =>
                  val from  = if (args.v.isEmpty) null else castTpeV(args.tpe, args.v)
                  val until = if (args.v2.isEmpty) null else castTpeV(args.tpe2, args.v2)
                  conn.peerConn.log.txRange(from, until).asScala.foldLeft(Future(new StringBuffer())) {
                    case (sbFut, txMap) =>
                      // Flatten transaction datoms to uniform tuples return type
                      val datom2packed = getPeerDatom2packed(Some(txMap.get(datomic.Log.T).asInstanceOf[Long]))
                      txMap.get(datomic.Log.DATA).asInstanceOf[jList[PeerDatom]].asScala.foldLeft(sbFut) {
                        case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                      }
                  }

                case conn: Conn_Client =>
                  val from  = if (args.v.isEmpty) None else Some(castTpeV(args.tpe, args.v))
                  val until = if (args.v2.isEmpty) None else Some(castTpeV(args.tpe2, args.v2))
                  conn.clientConn.txRange(from, until).foldLeft(Future(new StringBuffer())) {
                    case (sbFut, (t, datoms)) =>
                      // Flatten transaction datoms to uniform tuples return type
                      val datom2packed: (StringBuffer, ClientDatom) => Future[StringBuffer] =
                        getClientDatom2packed(Some(t))
                      datoms.foldLeft(sbFut) {
                        case (sbFut, datom) => sbFut.flatMap(sb => datom2packed(sb, datom))
                      }
                  }
              }
          }

          sbFut.map(_.toString)
        }
      } yield {
        //        println("-------------------------------" + packed)
        packed
      }
    } catch {
      case NonFatal(exc) => Future.failed(exc)
    }
  }


  // Presuming a datalog query returning rows of single values.
  // Card-many attributes should therefore not be returned as Sets.
  def getAttrValues(
    connProxy: ConnProxy,
    datalogQuery: String,
    card: Int,
    tpe: String
  ): Future[List[String]] = {
    for {
      conn <- getConn(connProxy)
      rows0 <- conn.rawQuery(datalogQuery)
    } yield {
      val cast = if (tpe == "Date" && card != 3)
        (v: Any) => date2str(v.asInstanceOf[Date])
      else
        (v: Any) => v.toString
      var vs   = List.empty[String]
      rows0.forEach(row => vs = vs :+ cast(row.get(0)))
      vs
    }
  }

  def getEntityAttrKeys(
    connProxy: ConnProxy,
    query: String
  ): Future[List[String]] = {
    var list = List.empty[String]
    for {
      conn <- getConn(connProxy)
      rows <- conn.rawQuery(query)
    } yield {
      rows.forEach { row =>
        list = row.get(0).toString :: list
      }
      list.sorted
    }
  }


  def basisT(connProxy: ConnProxy): Future[Long] = {
    for {
      conn <- getConn(connProxy)
      db <- conn.db
      t <- db.basisT
    } yield t
  }


  def retract(
    connProxy: ConnProxy,
    stmtsEdn: String,
    uriAttrs: Set[String]
  ): Future[TxReport] = {
    println(stmtsEdn)
    for {
      conn <- getConn(connProxy)
      javaStmts = getJavaStmts(stmtsEdn, uriAttrs)
      txReport <- conn.transact(javaStmts)
    } yield TxReportRPC(
      txReport.t, txReport.tx, txReport.txInstant, txReport.eids, txReport.txData, txReport.toString
    )
  }


  // Entity api ---------------------------------------------------

  def rawValue(connProxy: ConnProxy, eid: Long, attr: String): Future[String] = {
    getDatomicEntity(connProxy, eid)
      .flatMap(_.rawValue(attr))
      .map(res => entityList2packed(List(attr -> res)))
  }

  def asMap(connProxy: ConnProxy, eid: Long, depth: Int, maxDepth: Int): Future[String] = {
    getDatomicEntity(connProxy, eid).flatMap(_.asMap(depth, maxDepth)).map(entityMap2packed)
  }

  def asList(connProxy: ConnProxy, eid: Long, depth: Int, maxDepth: Int): Future[String] = {
    getDatomicEntity(connProxy, eid).flatMap(_.asList(depth, maxDepth)).map(entityList2packed)
  }


  def attrs(connProxy: ConnProxy, eid: Long): Future[List[String]] = {
    getDatomicEntity(connProxy, eid).flatMap(_.attrs)
  }


  def apply(connProxy: ConnProxy, eid: Long, attr: String): Future[String] = {
    getDatomicEntity(connProxy, eid)
      .flatMap(_.apply[Any](attr))
      .map(_.fold("")(v => entityMap2packed(Map(attr -> v))))
  }

  def apply(connProxy: ConnProxy, eid: Long, attrs: List[String]): Future[List[String]] = {
    val attr1 :: attr2 :: moreAttrs = attrs
    getDatomicEntity(connProxy, eid)
      .flatMap(_.apply(attr1, attr2, moreAttrs: _*))
      .map { optValues =>
        attrs.zip(optValues).map { case (attr, optV) =>
          optV.fold("")(v => entityMap2packed(Map(attr -> v)))
        }
      }
  }

  def graphDepth(connProxy: ConnProxy, eid: Long, maxDepth: Int): Future[String] = {
    // Use list to guarantee order of attributes for packing
    getDatomicEntity(connProxy, eid).flatMap(_.asList(1, maxDepth)).map(entityList2packed)
  }

  def graphCode(connProxy: ConnProxy, eid: Long, maxDepth: Int): Future[String] = {
    getDatomicEntity(connProxy, eid).flatMap(_.graphCode(maxDepth))
  }

  private def getDatomicEntity(connProxy: ConnProxy, eid: Any): Future[DatomicEntity] = {
    for {
      conn <- getConn(connProxy)
      db <- conn.db
    } yield db.entity(conn, eid)
  }


  // Connection pool ---------------------------------------------

  // todo - this is primitive, is a more correct implementation needed?
  private val connectionPool = mutable.HashMap.empty[String, Future[Conn]]

  def clearConnPool: Future[Unit] = Future {
    //    println(s"Connection pool with ${connectionPool.size} connections cleared.")
    connectionPool.clear()
  }

  private def getFreshConn(connProxy: ConnProxy): Future[Conn] = connProxy match {
    case proxy@DatomicPeerProxy(protocol, dbIdentifier, schema, _, _, _, _, _) =>
      protocol match {
        case "mem" =>
          Datomic_Peer.recreateDbFromEdn(proxy, schema)
            .recoverWith { case exc => Future.failed[Conn](MoleculeException(exc.getMessage)) }

        case "free" | "dev" | "pro" =>
          Datomic_Peer.connect(proxy, protocol, dbIdentifier)
            .recoverWith { case exc => Future.failed[Conn](MoleculeException(exc.getMessage)) }

        case other =>
          Future.failed(MoleculeException(
            s"\nCan't serve Peer protocol `$other`."
          ))
      }

    case proxy@DatomicDevLocalProxy(protocol, system, storageDir, dbName, schema, _, _, _, _, _) =>
      val devLocal = Datomic_DevLocal(system, storageDir)
      protocol match {
        case "mem" =>
          devLocal.recreateDbFromEdn(schema, proxy)
            .recoverWith { case exc => Future.failed[Conn](MoleculeException(exc.getMessage)) }

        case "dev" | "pro" =>
          devLocal.connect(proxy, dbName)
            .recoverWith { case exc => Future.failed[Conn](MoleculeException(exc.getMessage)) }

        case other =>
          Future.failed(MoleculeException(
            s"\nCan't serve DevLocal protocol `$other`."
          ))
      }

    case proxy@DatomicPeerServerProxy(accessKey, secret, endpoint, dbName, _, _, _, _, _, _) =>
      Datomic_PeerServer(accessKey, secret, endpoint).connect(proxy, dbName)
        .recoverWith { case exc => Future.failed[Conn](MoleculeException(exc.getMessage)) }
  }

  private def getConn(
    connProxy: ConnProxy
  ): Future[Conn] = {
    val futConn             = connectionPool.getOrElse(connProxy.uuid, getFreshConn(connProxy))
    val futConnTimeAdjusted = futConn.map { conn =>
      conn.updateAdhocDbView(connProxy.adhocDbView)
      conn.updateTestDbView(connProxy.testDbView, connProxy.testDbStatus)
      conn
    }
    connectionPool(connProxy.uuid) = futConnTimeAdjusted
    futConnTimeAdjusted
  }


  // Helpers -------------------------------------------------

  def getJavaStmts(
    stmtsEdn: String,
    uriAttrs: Set[String]
  ): jList[AnyRef] = {
    val stmts = readAll(new StringReader(stmtsEdn)).get(0).asInstanceOf[jList[AnyRef]]
    if (uriAttrs.isEmpty) {
      stmts
    } else {
      def uri(s: AnyRef): AnyRef = readString(s"""#=(new java.net.URI "$s")""")
      val stmtsSize = stmts.size()
      val newStmts  = new util.ArrayList[jList[_]](stmtsSize)
      stmts.forEach { stmtRaw =>
        val stmt = stmtRaw.asInstanceOf[jList[AnyRef]]
        if (uriAttrs.contains(stmt.get(2).toString)) {
          val uriStmt = stmt.get(0).toString match {
            case ":db/add"     => list(stmt.get(0), stmt.get(1), stmt.get(2), uri(stmt.get(3)))
            case ":db/retract" => list(stmt.get(0), stmt.get(1), stmt.get(2), uri(stmt.get(3)))
            case ":db.fn/cas"  => list(stmt.get(0), stmt.get(1), stmt.get(2), uri(stmt.get(3)), uri(stmt.get(4)))
            case _             => stmt
          }
          newStmts.add(uriStmt)
        } else {
          newStmts.add(stmt)
        }
      }
      Collections.unmodifiableList(newStmts)
    }
  }

  def qTime(queryTime: Long): String = {
    val indents = 5 - queryTime.toString.length
    " " * indents + thousands(queryTime) + " ms"
  }
}
