/*
 Copyright 2013 Jason Baldridge

 Licensed under the Apache License, Version 2.0 (the "License")
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/
package nak.cluster

/**
 * A confusion matrix for comparing gold clusters to some predicted clusters.
 *
 * @param goldLabels	the set of labels from the gold labeled data
 * @param predictedOutcomes a set of (usually autogenerated) labels for the predicted clusters
 * @param counts the matrix, where each cell is the number of data points that had a given gold label and predicted label
 *
 * @author jasonbaldridge
 */
class ClusterConfusionMatrix(
  goldLabels: IndexedSeq[String],
  predictedOutcomes: IndexedSeq[String],
  counts: IndexedSeq[IndexedSeq[Int]]) {

  // Create a string representation. Be lazy so that we only do it once.
  lazy val stringRep = {
    val lengthOfRow = counts(0).mkString.length + counts(0).length * 7

    val tableString = counts.zip(goldLabels).map {
      case (goldLine, goldLabel) =>
        (goldLine.mkString("\t") +
          "\t|\t" + goldLine.sum + "\t[" + goldLabel + "]")
    }.mkString("\n")

    ("-" * 80 + "\n" + "Confusion matrix.\n" +
      "Columns give predicted counts. Rows give gold counts.\n" +
      "-" * 80 + "\n" +
      tableString + "\n" +
      "-" * lengthOfRow + "\n" +
      counts.transpose.map(_.sum).mkString("\t") + "\n" +
      predictedOutcomes.map("[" + _ + "]").mkString("\t") + "\n")
  }

  // Get the string representation.
  override def toString = stringRep
}

/**
 * A companion object for constructing ClusterConfusionMatrices.
 */
object ClusterConfusionMatrix {

  /**
   * Construct a confusion matrix for comparing gold clusters
   * to some predicted clusters.
   *
   * @param goldClusterIds	a sequence of cluster ids for each data point
   * @param numPredictedClusters	the number of clusters predicted
   * @param predictedClusterIndices	sequence of cluster indices (0 to numPredictedClusters-1) for each data point
   * @return				a ClusterConfusionMatrix with the relevant comparisions
   */
  def apply(
    goldClusterIds: IndexedSeq[String],
    numPredictedClusters: Int,
    predictedClusterIndices: IndexedSeq[Int]) = {

    val goldLabels = goldClusterIds.toSet.toIndexedSeq
    val goldIndices = goldLabels.zipWithIndex.toMap

    val numGoldClusters = goldIndices.size

    val counts = Array.fill(numGoldClusters, numPredictedClusters)(0)

    goldClusterIds.zip(predictedClusterIndices).map {
      case (goldId, predIndex) =>
        counts(goldIndices(goldId))(predIndex) += 1
    }

    new ClusterConfusionMatrix(
      goldLabels,
      (0 until numPredictedClusters).map(_.toString),
      counts.map(_.toIndexedSeq).toIndexedSeq)
  }

}
