/*
 * Decompiled with CFR 0.152.
 */
package org.scion.jpan.internal;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.scion.jpan.ScionException;
import org.scion.jpan.ScionRuntimeException;
import org.scion.jpan.ScionUtil;
import org.scion.jpan.internal.DNSHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScionBootstrapper {
    private static final Logger LOG = LoggerFactory.getLogger((String)ScionBootstrapper.class.getName());
    private static final String BASE_URL = "";
    private static final String TOPOLOGY_ENDPOINT = "topology";
    private static final Duration httpRequestTimeout = Duration.of(2L, ChronoUnit.SECONDS);
    private final String topologyResource;
    private final List<ServiceNode> controlServices = new ArrayList<ServiceNode>();
    private final List<ServiceNode> discoveryServices = new ArrayList<ServiceNode>();
    private final List<BorderRouter> borderRouters = new ArrayList<BorderRouter>();
    private String localIsdAs;
    private boolean isCoreAs;
    private int localMtu;

    protected ScionBootstrapper(String topologyServiceAddress) {
        this.topologyResource = topologyServiceAddress;
        this.init();
    }

    protected ScionBootstrapper(Path file) {
        this.topologyResource = file.toString();
        this.init(file);
    }

    public static synchronized ScionBootstrapper createViaDns(String host) {
        return new ScionBootstrapper(ScionBootstrapper.bootstrapViaDNS(host));
    }

    public static synchronized ScionBootstrapper createViaBootstrapServerIP(String hostAndPort) {
        return new ScionBootstrapper(hostAndPort);
    }

    public static synchronized ScionBootstrapper createViaTopoFile(Path file) {
        return new ScionBootstrapper(file);
    }

    private static String bootstrapViaDNS(String hostName) {
        try {
            String addr = DNSHelper.getScionDiscoveryAddress(hostName);
            if (addr == null) {
                throw new ScionRuntimeException("No valid DNS NAPTR entry found for host: " + hostName);
            }
            return addr;
        }
        catch (IOException e) {
            throw new ScionRuntimeException("Error while bootstrapping Scion via DNS: " + e.getMessage());
        }
    }

    private static JsonElement safeGet(JsonObject o, String name) {
        JsonElement e = o.get(name);
        if (e == null) {
            throw new ScionRuntimeException("Entry not found in topology file: " + name);
        }
        return e;
    }

    /*
     * Loose catch block
     */
    private static String fetchTopologyFile(URL url) throws IOException {
        HttpURLConnection httpURLConnection = null;
        try {
            httpURLConnection = (HttpURLConnection)url.openConnection();
            httpURLConnection.setRequestMethod("GET");
            httpURLConnection.setConnectTimeout((int)httpRequestTimeout.toMillis());
            int responseCode = httpURLConnection.getResponseCode();
            if (responseCode != 200) {
                throw new ScionException("GET request failed (" + responseCode + ") on topology server: " + url);
            }
            try (BufferedReader in = new BufferedReader(new InputStreamReader(httpURLConnection.getInputStream()));){
                String inputLine;
                StringBuilder response = new StringBuilder();
                while ((inputLine = in.readLine()) != null) {
                    response.append(inputLine).append(System.lineSeparator());
                }
                String string = response.toString();
                return string;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            if (httpURLConnection != null) {
                httpURLConnection.disconnect();
            }
        }
    }

    private void init() {
        try {
            this.parseTopologyFile(this.getTopologyFile());
        }
        catch (IOException e) {
            throw new ScionRuntimeException("Error while getting topology file from " + this.topologyResource + ": " + e.getMessage(), e);
        }
        if (this.controlServices.isEmpty()) {
            throw new ScionRuntimeException("No control servers found in topology provided by " + this.topologyResource);
        }
    }

    private void init(Path file) {
        try {
            ClassLoader classLoader;
            URL resource;
            if (!Files.exists(file, new LinkOption[0]) && (resource = (classLoader = this.getClass().getClassLoader()).getResource(file.toString())) != null) {
                file = Paths.get(resource.toURI());
            }
        }
        catch (URISyntaxException e) {
            throw new ScionRuntimeException(e);
        }
        StringBuilder contentBuilder = new StringBuilder();
        try (Stream<String> stream = Files.lines(file, StandardCharsets.UTF_8);){
            stream.forEach(s -> contentBuilder.append((String)s).append("\n"));
        }
        catch (IOException e) {
            throw new ScionRuntimeException("Error reading topology file: " + file.toAbsolutePath(), e);
        }
        this.parseTopologyFile(contentBuilder.toString());
        if (this.controlServices.isEmpty()) {
            throw new ScionRuntimeException("No control service found in topology filet: " + file);
        }
    }

    public String getControlServerAddress() {
        return this.controlServices.get((int)0).ipString;
    }

    public boolean isLocalAsCore() {
        return this.isCoreAs;
    }

    public long getLocalIsdAs() {
        return ScionUtil.parseIA(this.localIsdAs);
    }

    public String getBorderRouterAddress(int interfaceId) {
        for (BorderRouter br : this.borderRouters) {
            for (BorderRouterInterface brif : br.interfaces) {
                if (brif.id != interfaceId) continue;
                return br.internalAddress;
            }
        }
        throw new ScionRuntimeException("No router found with interface ID " + interfaceId);
    }

    public List<String> getBorderRouterAddresses() {
        ArrayList<String> result = new ArrayList<String>();
        for (BorderRouter br : this.borderRouters) {
            result.add(br.internalAddress);
        }
        return result;
    }

    public int getLocalMtu() {
        return this.localMtu;
    }

    public void refreshTopology() {
        throw new UnsupportedOperationException();
    }

    private String getTopologyFile() throws IOException {
        LOG.info("Getting topology from bootstrap server: {}", (Object)this.topologyResource);
        this.controlServices.clear();
        this.discoveryServices.clear();
        URL url = new URL("http://" + this.topologyResource + "/" + BASE_URL + TOPOLOGY_ENDPOINT);
        return ScionBootstrapper.fetchTopologyFile(url);
    }

    private void parseTopologyFile(String topologyFile) {
        JsonElement jsonTree = JsonParser.parseString((String)topologyFile);
        if (jsonTree.isJsonObject()) {
            JsonObject o = jsonTree.getAsJsonObject();
            this.localIsdAs = ScionBootstrapper.safeGet(o, "isd_as").getAsString();
            this.localMtu = ScionBootstrapper.safeGet(o, "mtu").getAsInt();
            JsonObject brs = ScionBootstrapper.safeGet(o, "border_routers").getAsJsonObject();
            for (Object e : brs.entrySet()) {
                JsonObject br = ((JsonElement)e.getValue()).getAsJsonObject();
                String addr = ScionBootstrapper.safeGet(br, "internal_addr").getAsString();
                JsonObject ints = ScionBootstrapper.safeGet(br, "interfaces").getAsJsonObject();
                ArrayList<BorderRouterInterface> interfaces = new ArrayList<BorderRouterInterface>();
                for (Map.Entry ifEntry : ints.entrySet()) {
                    JsonObject ife = ((JsonElement)ifEntry.getValue()).getAsJsonObject();
                    JsonObject underlay = ife.getAsJsonObject("underlay");
                    JsonElement local = underlay.has("local") ? underlay.get("local") : underlay.get("public");
                    JsonElement remote = underlay.get("remote");
                    interfaces.add(new BorderRouterInterface((String)ifEntry.getKey(), local.getAsString(), remote.getAsString()));
                }
                this.borderRouters.add(new BorderRouter((String)e.getKey(), addr, interfaces));
            }
            JsonObject css = ScionBootstrapper.safeGet(o, "control_service").getAsJsonObject();
            for (Object e : css.entrySet()) {
                JsonObject cs = ((JsonElement)e.getValue()).getAsJsonObject();
                this.controlServices.add(new ServiceNode((String)e.getKey(), cs.get("addr").getAsString()));
            }
            JsonObject dss = ScionBootstrapper.safeGet(o, "discovery_service").getAsJsonObject();
            for (Map.Entry e : dss.entrySet()) {
                JsonObject ds = ((JsonElement)e.getValue()).getAsJsonObject();
                this.discoveryServices.add(new ServiceNode((String)e.getKey(), ds.get("addr").getAsString()));
            }
            JsonArray attr = ScionBootstrapper.safeGet(o, "attributes").getAsJsonArray();
            for (int i = 0; i < attr.size(); ++i) {
                if (!"core".equals(attr.get(i).getAsString())) continue;
                this.isCoreAs = true;
            }
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Topo Server: ").append(this.topologyResource).append("\n");
        sb.append("ISD/AS: ").append(this.localIsdAs).append('\n');
        sb.append("Core: ").append(this.isCoreAs).append('\n');
        sb.append("MTU: ").append(this.localMtu).append('\n');
        for (ServiceNode sn : this.controlServices) {
            sb.append("Control server:   ").append(sn).append('\n');
        }
        for (ServiceNode sn : this.discoveryServices) {
            sb.append("Discovery server: ").append(sn).append('\n');
        }
        return sb.toString();
    }

    private static class ServiceNode {
        final String name;
        final String ipString;

        ServiceNode(String name, String ipString) {
            this.name = name;
            this.ipString = ipString;
        }

        public String toString() {
            return "{name='" + this.name + '\'' + ", ipString='" + this.ipString + '\'' + '}';
        }
    }

    private static class BorderRouterInterface {
        final int id;
        final String publicUnderlay;
        final String remoteUnderlay;

        public BorderRouterInterface(String id, String publicU, String remoteU) {
            this.id = Integer.parseInt(id);
            this.publicUnderlay = publicU;
            this.remoteUnderlay = remoteU;
        }
    }

    private static class BorderRouter {
        private final String name;
        private final String internalAddress;
        private final List<BorderRouterInterface> interfaces;

        public BorderRouter(String name, String addr, List<BorderRouterInterface> interfaces) {
            this.name = name;
            this.internalAddress = addr;
            this.interfaces = interfaces;
        }
    }
}

