/*
 * Decompiled with CFR 0.152.
 */
package org.scion.jpan.internal;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;
import org.scion.jpan.ScionRuntimeException;
import org.scion.jpan.ScionUtil;
import org.scion.jpan.internal.IPHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HostsFileParser {
    private static final Logger LOG = LoggerFactory.getLogger(HostsFileParser.class);
    private static final String PATH_LINUX = "/etc/scion/hosts;/etc/hosts";
    private final String propHostsFiles = ScionUtil.getPropertyOrEnv("org.scion.hostsFiles", "SCION_HOSTS_FILES");
    private final Map<String, HostEntry> entries = new HashMap<String, HostEntry>();

    public HostsFileParser() {
        this.init();
    }

    private void init() {
        String os = System.getProperty("os.name");
        String hostsFiles = this.propHostsFiles != null && !this.propHostsFiles.isEmpty() ? this.propHostsFiles : ("Linux".equals(os) || "Mac OS X".equals(os) ? PATH_LINUX : "");
        LOG.debug("OS={}; hostsFiles=\"{}\"", (Object)os, (Object)hostsFiles);
        for (String file : hostsFiles.split(";")) {
            Path path = Paths.get(file, new String[0]);
            if (!Files.exists(path, new LinkOption[0]) || !Files.isRegularFile(path, new LinkOption[0])) {
                LOG.info("File not found or not accessible: {}", (Object)path);
                return;
            }
            try (Stream<String> lines = Files.lines(path);){
                lines.forEach(line -> this.parseLine((String)line, path));
            }
            catch (IOException e) {
                throw new ScionRuntimeException(e);
            }
        }
    }

    private void parseLine(String line, Path path) {
        try {
            String hostName;
            String s = line.trim();
            if (s.isEmpty() || s.startsWith("#")) {
                return;
            }
            String[] lineParts = s.split("\\s+");
            HostsFileParser.check(lineParts.length >= 2, "Expected ` `");
            String[] addrParts = lineParts[0].split(",");
            HostsFileParser.check(addrParts.length == 2, "Expected `,`");
            long isdIa = ScionUtil.parseIA(addrParts[0]);
            HostsFileParser.check(addrParts[1].startsWith("["), "Expected `[` before address");
            HostsFileParser.check(addrParts[1].endsWith("]"), "Expected `]` after address");
            String addrStr = addrParts[1].substring(1, addrParts[1].length() - 1).trim();
            HostsFileParser.check(!addrStr.isEmpty(), "Address is empty");
            byte[] addrBytes = IPHelper.toByteArray(addrStr);
            HostsFileParser.check(addrBytes != null, "Address string is not a legal address");
            for (int i = 1; i < lineParts.length && !(hostName = lineParts[i]).startsWith("#"); ++i) {
                InetAddress inetAddr = InetAddress.getByAddress(hostName, addrBytes);
                this.entries.put(hostName, new HostEntry(isdIa, inetAddr));
            }
            InetAddress inetAddr = InetAddress.getByAddress(addrStr, addrBytes);
            this.entries.put(addrStr, new HostEntry(isdIa, inetAddr));
            this.entries.put(inetAddr.getHostAddress(), new HostEntry(isdIa, inetAddr));
        }
        catch (IllegalArgumentException | IndexOutOfBoundsException | UnknownHostException e) {
            LOG.info("ERROR parsing file {}: error=\"{}\" line=\"{}\"", new Object[]{path, e.getMessage(), line});
        }
    }

    private static void check(boolean pass, String msg) {
        if (!pass) {
            throw new IllegalArgumentException(msg);
        }
    }

    public HostEntry find(String hostName) {
        return this.entries.get(hostName);
    }

    public static class HostEntry {
        private final long isdAs;
        private final InetAddress address;

        HostEntry(long isdAs, InetAddress address) {
            this.isdAs = isdAs;
            this.address = address;
        }

        public long getIsdAs() {
            return this.isdAs;
        }

        public InetAddress getAddress() {
            return this.address;
        }
    }
}

