/*
 * Decompiled with CFR 0.152.
 */
package org.scion.jpan.internal;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.spi.AbstractSelector;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.scion.jpan.Path;
import org.scion.jpan.ScionRuntimeException;
import org.scion.jpan.internal.ByteUtil;
import org.scion.jpan.internal.Config;
import org.scion.jpan.internal.ExternalIpDiscovery;
import org.scion.jpan.internal.IPHelper;
import org.scion.jpan.internal.STUN;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NatMapping {
    private static final Logger log = LoggerFactory.getLogger(NatMapping.class);
    private final long localIsdAs;
    private NatMode mode;
    private Entry commonAddress;
    private final Map<InetSocketAddress, Entry> sourceIPs = new HashMap<InetSocketAddress, Entry>();
    private final ByteBuffer buffer = ByteBuffer.allocateDirect(100);
    private final DatagramChannel channel;
    private InetAddress externalIP;
    private final Timer timer;
    private final int natMappingTimeoutMs = Config.getNatMappingTimeoutMs();
    private final int stunTimeoutMs = Config.getStunTimeoutMs();

    private NatMapping(DatagramChannel channel, long localIsdAs, List<InetSocketAddress> borderRouters) {
        this.channel = channel;
        this.localIsdAs = localIsdAs;
        this.mode = NatMode.NOT_INITIALIZED;
        this.timer = new Timer();
        boolean useTimer = Config.useNatMappingKeepAlive();
        for (InetSocketAddress brAddress : borderRouters) {
            this.sourceIPs.computeIfAbsent(brAddress, k -> {
                Entry e = new Entry(null, brAddress);
                if (useTimer) {
                    this.timer.schedule((TimerTask)new NatMappingTimerTask(e), this.natMappingTimeoutMs);
                }
                return e;
            });
        }
    }

    public synchronized void touch(InetSocketAddress borderRouterAddress) {
        Entry e = this.sourceIPs.get(borderRouterAddress);
        if (e == null) {
            log.info("No border router found for {}", (Object)borderRouterAddress);
            return;
        }
        e.touch();
    }

    private void update(NatMode mode, InetSocketAddress address) {
        this.mode = mode;
        if (this.commonAddress == null) {
            this.commonAddress = new Entry(address, null);
        } else {
            this.commonAddress.updateSource(address);
        }
    }

    public synchronized InetSocketAddress getMappedAddress(Path path) {
        switch (this.mode) {
            case NO_NAT: {
                return this.commonAddress.getMappedSource();
            }
            case STUN_SERVER: {
                if (this.commonAddress.isExpired(this.mode, this.natMappingTimeoutMs)) {
                    InetSocketAddress address = this.tryCustomServer(true);
                    if (address == null) {
                        throw new ScionRuntimeException("Failed to connect to STUN servers: \"" + Config.getNatStunServer() + "\"");
                    }
                    this.commonAddress.updateSource(address);
                }
                return this.commonAddress.getMappedSource();
            }
            case BR_STUN: {
                return this.getMappedAddressFromBR(path);
            }
        }
        throw new IllegalStateException();
    }

    public InetSocketAddress getMappedAddressFromBR(Path path) {
        Entry entry = this.sourceIPs.get(path.getFirstHopAddress());
        if (entry == null) {
            if (path.getRemoteIsdAs() == this.localIsdAs) {
                return this.commonAddress.getMappedSource();
            }
            throw new IllegalArgumentException("Unknown border router: " + path.getFirstHopAddress());
        }
        if (entry.getSource() == null || entry.isExpired(this.mode, this.natMappingTimeoutMs)) {
            try {
                entry.updateSource(this.tryStunAddress(entry.firstHop));
            }
            catch (IOException e) {
                throw new ScionRuntimeException(e);
            }
            if (entry.getSource() == null) {
                throw new IllegalStateException("No mapped source for: " + path.getFirstHopAddress());
            }
        } else {
            entry.touch();
        }
        return entry.getSource();
    }

    public synchronized InetAddress getExternalIP() {
        if (this.externalIP == null) {
            try {
                InetSocketAddress local = (InetSocketAddress)this.channel.getLocalAddress();
                this.externalIP = local.getAddress();
                if (this.externalIP.isAnyLocalAddress()) {
                    InetSocketAddress firstHop = this.sourceIPs.values().iterator().next().firstHop;
                    this.externalIP = ExternalIpDiscovery.getExternalIP(firstHop);
                }
            }
            catch (IOException e) {
                throw new ScionRuntimeException(e);
            }
        }
        return this.externalIP;
    }

    private static ConfigMode getConfig() {
        switch (Config.getNat()) {
            case "OFF": {
                return ConfigMode.STUN_OFF;
            }
            case "BR": {
                return ConfigMode.STUN_BR;
            }
            case "CUSTOM": {
                return ConfigMode.STUN_CUSTOM;
            }
            case "AUTO": {
                return ConfigMode.STUN_AUTO;
            }
        }
        throw new IllegalArgumentException("Illegal value for NAT config: \"" + Config.getNat() + "\"");
    }

    public static NatMapping createMapping(long localIsdAs, DatagramChannel channel, List<InetSocketAddress> borderRouters) {
        if (borderRouters.isEmpty()) {
            log.warn("No border routers found in local topology information.");
        }
        NatMapping natMapping = new NatMapping(channel, localIsdAs, borderRouters);
        try {
            natMapping.detectSourceAddress();
        }
        catch (IOException e) {
            throw new ScionRuntimeException(e);
        }
        return natMapping;
    }

    private void detectSourceAddress() throws IOException {
        ConfigMode configMode = NatMapping.getConfig();
        switch (configMode) {
            case STUN_OFF: {
                this.update(NatMode.NO_NAT, this.getLocalAddress());
                break;
            }
            case STUN_AUTO: {
                this.autoDetect();
                break;
            }
            case STUN_CUSTOM: {
                InetSocketAddress address = this.tryCustomServer(true);
                if (address == null) {
                    String custom = Config.getNatStunServer();
                    throw new ScionRuntimeException("Failed to connect to STUN servers: \"" + custom + "\"");
                }
                this.update(NatMode.STUN_SERVER, address);
                break;
            }
            case STUN_BR: {
                this.tryStunBorderRouters();
                this.update(NatMode.BR_STUN, this.getLocalAddress());
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown config mode: " + (Object)((Object)configMode));
            }
        }
    }

    private InetSocketAddress getLocalAddress() throws IOException {
        InetSocketAddress local = (InetSocketAddress)this.channel.getLocalAddress();
        InetAddress localIP = local.getAddress();
        if (localIP.isAnyLocalAddress()) {
            localIP = this.getExternalIP();
            local = new InetSocketAddress(localIP, local.getPort());
        }
        return local;
    }

    private InetSocketAddress tryCustomServer(boolean throwOnFailure) {
        String custom = Config.getNatStunServer();
        if (!throwOnFailure && (custom == null || custom.isEmpty())) {
            return null;
        }
        return this.tryStunServer(custom);
    }

    private void autoDetect() throws IOException {
        InetSocketAddress source = this.tryCustomServer(false);
        if (source != null) {
            this.update(NatMode.STUN_SERVER, source);
            log.info("NAT AUTO: Found custom STUN server.");
            return;
        }
        if (this.tryStunBorderRouters()) {
            this.update(NatMode.BR_STUN, this.getLocalAddress());
            log.info("NAT AUTO: Found STUN enabled border routers.");
            return;
        }
        int localPort = ((InetSocketAddress)this.channel.getLocalAddress()).getPort();
        InetAddress sourceIP = this.getExternalIP();
        source = new InetSocketAddress(sourceIP, localPort);
        this.update(NatMode.NO_NAT, source);
        log.info("Could not find custom STUN server or NAT enabled border routers. Hoping for no NAT.");
    }

    private InetSocketAddress tryStunServer(String stunAddress) {
        String[] servers;
        for (String server : servers = stunAddress.split(";")) {
            InetSocketAddress stunServer;
            try {
                stunServer = IPHelper.toInetSocketAddress(server);
            }
            catch (IllegalArgumentException e) {
                String prefix = "Please provide a valid STUN server address as 'address:port' in SCION_STUN_SERVER or org.scion.stun.server, was: ";
                log.error("{}\"{}\"", (Object)prefix, (Object)server);
                throw new IllegalArgumentException(prefix + "\"" + server + "\"");
            }
            try {
                InetSocketAddress address = this.tryStunAddress(stunServer);
                if (address == null) continue;
                return address;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return null;
    }

    /*
     * Loose catch block
     */
    private boolean tryStunBorderRouters() throws IOException {
        boolean isBlocking = this.channel.isBlocking();
        try {
            try (AbstractSelector selector = this.channel.provider().openSelector();){
                this.channel.configureBlocking(false);
                this.channel.register(selector, 1, this.channel);
                boolean bl = this.doStunRequest(selector, this.sourceIPs.values());
                return bl;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.channel.configureBlocking(isBlocking);
        }
    }

    /*
     * Loose catch block
     */
    private InetSocketAddress tryStunAddress(InetSocketAddress server) throws IOException {
        boolean isBlocking = this.channel.isBlocking();
        try {
            try (AbstractSelector selector = this.channel.provider().openSelector();){
                this.channel.configureBlocking(false);
                this.channel.register(selector, 1, this.channel);
                ArrayList<Entry> servers = new ArrayList<Entry>();
                servers.add(Entry.createForStunServer(server));
                if (this.doStunRequest(selector, servers)) {
                    InetSocketAddress inetSocketAddress = ((Entry)servers.get(0)).getSource();
                    return inetSocketAddress;
                }
                InetSocketAddress inetSocketAddress = null;
                return inetSocketAddress;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            this.channel.configureBlocking(isBlocking);
        }
    }

    private boolean doStunRequest(Selector selector, Collection<Entry> servers) throws IOException {
        HashMap<STUN.TransactionID, Entry> ids = new HashMap<STUN.TransactionID, Entry>();
        for (Entry e : servers) {
            this.buffer.clear();
            STUN.TransactionID id = STUN.writeRequest(this.buffer);
            ids.put(id, e);
            this.buffer.flip();
            this.channel.send(this.buffer, e.firstHop);
        }
        while (selector.select(this.stunTimeoutMs) > 0) {
            Iterator<SelectionKey> iter = selector.selectedKeys().iterator();
            while (iter.hasNext()) {
                SelectionKey key = iter.next();
                iter.remove();
                if (!key.isReadable()) continue;
                DatagramChannel channelIn = (DatagramChannel)key.channel();
                this.buffer.clear();
                channelIn.receive(this.buffer);
                this.buffer.flip();
                ByteUtil.MutRef<String> error = new ByteUtil.MutRef<String>();
                ByteUtil.MutRef<STUN.TransactionID> id = new ByteUtil.MutRef<STUN.TransactionID>();
                InetSocketAddress external = STUN.parseResponse(this.buffer, ids::containsKey, id, error);
                Entry e = (Entry)ids.remove(id.get());
                if (e == null) continue;
                e.updateSource(external);
                if (external == null || error.get() != null || !ids.isEmpty()) continue;
                return true;
            }
        }
        return false;
    }

    public void close() {
        this.timer.cancel();
    }

    private class NatMappingTimerTask
    extends TimerTask {
        private final Entry e;
        private final Exception e2;

        NatMappingTimerTask(Entry e) {
            this.e = e;
            this.e2 = new RuntimeException();
        }

        NatMappingTimerTask(Entry e, Exception e2) {
            this.e = e;
            this.e2 = e2;
        }

        @Override
        public void run() {
            long nextRequiredUse = this.e.lastUsed + (long)NatMapping.this.natMappingTimeoutMs;
            long delay = nextRequiredUse - System.currentTimeMillis();
            if (delay <= 0L) {
                this.e.touch();
                ByteBuffer buf = ByteBuffer.allocate(100);
                STUN.writeRequest(buf);
                buf.flip();
                try {
                    NatMapping.this.channel.send(buf, this.e.firstHop);
                }
                catch (IOException ex) {
                    log.error("Error while sending keep alive to {}", (Object)this.e.firstHop, (Object)ex);
                }
                delay = NatMapping.this.natMappingTimeoutMs;
            }
            NatMapping.this.timer.schedule((TimerTask)new NatMappingTimerTask(this.e, this.e2), delay);
        }
    }

    private static class Entry {
        private InetSocketAddress source;
        private final InetSocketAddress firstHop;
        private long lastUsed;

        Entry(InetSocketAddress source, InetSocketAddress firstHop) {
            this.source = source;
            this.firstHop = firstHop;
            this.touch();
        }

        static Entry createForStunServer(InetSocketAddress server) {
            return new Entry(null, server);
        }

        InetSocketAddress getSource() {
            return this.source;
        }

        private void updateSource(InetSocketAddress source) {
            this.touch();
            this.source = source;
        }

        private InetSocketAddress getMappedSource() {
            this.touch();
            return this.source;
        }

        private void touch() {
            this.lastUsed = System.currentTimeMillis();
        }

        private boolean isExpired(NatMode mode, int natMappingTimeoutMs) {
            return mode != NatMode.NO_NAT && System.currentTimeMillis() - this.lastUsed > (long)natMappingTimeoutMs;
        }
    }

    private static enum NatMode {
        NOT_INITIALIZED,
        BR_STUN,
        NO_NAT,
        STUN_SERVER;

    }

    private static enum ConfigMode {
        STUN_OFF,
        STUN_BR,
        STUN_CUSTOM,
        STUN_AUTO;

    }
}

