/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core;

import static java.util.Locale.ROOT;

/**
 * Operations with strings.
 *
 * @since 1.0
 */
public class Strings {

	private Strings() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns the given string if it is non-empty, {@code null} otherwise.
	 *
	 * @since 1.0
	 */
	public static String emptyToNull(String string) {
		return isNullOrEmpty(string) ? null : string;
	}

	/**
	 * Checks whether the given string ends with the given suffix, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code suffix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean endsWith(String string, String suffix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(suffix != null, "Suffix must not be null");

		return string.regionMatches(ignoreCase, string.length() - suffix.length(), suffix, 0, suffix.length());
	}

	/**
	 * Ensures that the given string starts with the given prefix.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code prefix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String ensurePrefix(String string, String prefix) {
		return ensurePrefix(string, prefix, false);
	}

	/**
	 * Ensures that the given string starts with the given prefix, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code prefix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String ensurePrefix(String string, String prefix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(prefix != null, "Prefix must not be null");

		if (!startsWith(string, prefix, ignoreCase)) {
			return prefix + string;
		} else {
			return string;
		}
	}

	/**
	 * Ensures that the given string ends with the given suffix.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code suffix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String ensureSuffix(String string, String suffix) {
		return ensureSuffix(string, suffix, false);
	}

	/**
	 * Ensures that the given string ends with the given suffix, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code suffix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String ensureSuffix(String string, String suffix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(suffix != null, "Suffix must not be null");

		if (!endsWith(string, suffix, ignoreCase)) {
			return string + suffix;
		} else {
			return string;
		}
	}

	/**
	 * Checks whether the given string is {@code null} or empty.
	 *
	 * @since 1.0
	 */
	public static boolean isNullOrEmpty(String string) {
		return (string == null) || string.isEmpty();
	}

	/**
	 * Removes any leading and trailing whitespace characters from the given string and replaces any contained sequences of whitespace characters by a single space.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String normalizeWhitespace(String string) {
		Contract.checkArgument(string != null, "string must not be null");

		String result = string.trim();
		result = result.replaceAll("\\p{Space}+", " ");

		return result;
	}

	/**
	 * Returns the given string if it is non-empty, an empty string otherwise.
	 *
	 * @since 1.0
	 */
	public static String nullToEmpty(String string) {
		return (string == null) ? "" : string;
	}

	/**
	 * Removes the given prefix from the given string if present.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code prefix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String removePrefix(String string, String prefix) {
		return removePrefix(string, prefix, false);
	}

	/**
	 * Removes the given prefix from the given string if present, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code prefix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String removePrefix(String string, String prefix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(prefix != null, "Prefix must not be null");

		int prefixPosition = 0;
		if (string.regionMatches(ignoreCase, prefixPosition, prefix, 0, prefix.length())) {
			return string.substring(prefix.length());
		} else {
			return string;
		}
	}

	/**
	 * Removes the given suffix from the given string if present.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code suffix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String removeSuffix(String string, String suffix) {
		return removeSuffix(string, suffix, false);
	}

	/**
	 * Removes the given suffix from the given string if present, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code suffix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String removeSuffix(String string, String suffix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(suffix != null, "Suffix must not be null");

		int suffixPosition = string.length() - suffix.length();
		if (string.regionMatches(ignoreCase, suffixPosition, suffix, 0, suffix.length())) {
			return string.substring(0, suffixPosition);
		} else {
			return string;
		}
	}

	/**
	 * Removes all whitespace characters from the given string.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String removeWhitespace(String string) {
		Contract.checkArgument(string != null, "string must not be null");

		return string.replaceAll("\\p{Space}+", "");
	}

	/**
	 * Checks whether the given string starts with the given prefix, possibly ignoring case.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 * @throws IllegalArgumentException if {@code prefix} is {@code null}
	 *
	 * @since 1.0
	 */
	public static boolean startsWith(String string, String prefix, boolean ignoreCase) {
		Contract.checkArgument(string != null, "String must not be null");
		Contract.checkArgument(prefix != null, "Prefix must not be null");

		return string.regionMatches(ignoreCase, 0, prefix, 0, prefix.length());
	}

	/**
	 * Returns the given string converted to lower case in a locale-insensitive way.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String toLowerCase(String string) {
		Contract.checkArgument(string != null, "String must not be null");

		return string.toLowerCase(ROOT);
	}

	/**
	 * Returns the given string converted to upper case in a locale-insensitive way.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String toUpperCase(String string) {
		Contract.checkArgument(string != null, "String must not be null");

		return string.toUpperCase(ROOT);
	}

}
