/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core.io;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.sellcom.core.Contract;

/**
 * Operations with file system paths.
 *
 * @since 1.0
 */
public class MorePaths {

	private MorePaths() {
		// Utility class, not to be instantiated
	}


	/**
	 * Returns the given path in the canonical (normalized absolute) form.
	 *
	 * @throws IllegalArgumentException if {@code path} is {@code null}
	 *
	 * @since 1.0
	 */
	public static Path canonicalize(Path path) {
		Contract.checkArgument(path != null, "Path must not be null");

		// NOTE: The order of the operations is important to correctly handle
		// situations like in this method call:
		//
		//     Paths2.canonicalize(Paths.get("../some_file"));

		return path.toAbsolutePath().normalize();
	}

	/**
	 * Returns the extension of the file represented by the given path.
	 *
	 * @throws IllegalArgumentException if {@code path} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String getFileExtension(Path path) {
		String fileName = getFileName(path);

		int lastDot = fileName.lastIndexOf('.');

		return (lastDot > 0) ? fileName.substring(lastDot + 1) : "";
	}

	/**
	 * Returns the name of the file represented by the given path.
	 *
	 * @throws IllegalArgumentException if {@code path} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String getFileName(Path path) {
		Contract.checkArgument(path != null, "Path must not be null");

		Path fileName = path.getFileName();

		return (fileName != null) ? fileName.toString() : "";
	}

	/**
	 * Returns the name of the file represented by the given path without extension.
	 *
	 * @throws IllegalArgumentException if {@code path} is {@code null}
	 *
	 * @since 1.0
	 */
	public static String getFileNameWithoutExtension(Path path) {
		String fileName = getFileName(path);

		int lastDot = fileName.lastIndexOf('.');

		return (lastDot > 0) ? fileName.substring(0, lastDot) : fileName;
	}

	/**
	 * Returns the home directory of the current JRE.
	 *
	 * @since 1.0
	 */
	public static Path getJavaHomeDirectory() {
		return Paths.get(System.getProperty("java.home"));
	}

	/**
	 * Returns the home directory of the current user.
	 *
	 * @since 1.0
	 */
	public static Path getUserHomeDirectory() {
		return Paths.get(System.getProperty("user.home"));
	}

	/**
	 * Returns the working directory.
	 *
	 * @since 1.0
	 */
	public static Path getWorkingDirectory() {
		return Paths.get("");
	}

}
