/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core.net;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;

/**
 * Operations with IP addresses.
 *
 * @since 1.0
 */
public class InetAddresses {

	private InetAddresses() {
		// Utility class, not to be instantiated
	}


	/**
	 * Checks whether the given string represents an IPv4 address.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null} or empty
	 *
	 * @since 1.0
	 */
	public static boolean isInet4Address(String address) {
		Contract.checkArgument(!Strings.isNullOrEmpty(address), "Address must not be null or empty");

		try {
			InetAddress parsedAddress = InetAddress.getByName(address);

			return (parsedAddress instanceof Inet4Address);
		} catch (UnknownHostException e) {
			return false;
		}
	}

	/**
	 * Checks whether the given string represents an IPv6 address.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null} or empty
	 *
	 * @since 1.0
	 */
	public static boolean isInet6Address(String address) {
		Contract.checkArgument(!Strings.isNullOrEmpty(address), "Address must not be null or empty");

		try {
			InetAddress parsedAddress = InetAddress.getByName(address);

			return (parsedAddress instanceof Inet6Address);
		} catch (UnknownHostException e) {
			return false;
		}
	}

}
