/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core.i18n;

import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;

/**
 * {@link PropertyResourceBundle} loading resources from a UTF-8 encoded properties file.
 *
 * @since 1.0
 */
public class Utf8ResourceBundle extends PropertyResourceBundle {

	private static final String SUFFIX = "utf8";


	private Utf8ResourceBundle(Reader reader) throws IOException {
		super(reader);
	}


	/**
	 * Returns a {@code Utf8ResourceBundle} located using the given base name, the default locale, and the caller's class loader.
	 *
	 * @throws IllegalArgumentException if {@code baseName} is {@code null} or empty
	 *
	 * @see ResourceBundle#getBundle(String)
	 *
	 * @since 1.0
	 */
	public static Utf8ResourceBundle getUtf8Bundle(String baseName) {
		return getUtf8Bundle(baseName, Locale.getDefault());
	}

	/**
	 * Returns a {@code Utf8ResourceBundle} located using the given base name and locale, and the caller's class loader.
	 *
	 * @throws IllegalArgumentException if {@code baseName} is {@code null} or empty
	 * @throws IllegalArgumentException if {@code locale} is {@code null}
	 *
	 * @see ResourceBundle#getBundle(String, Locale)
	 *
	 * @since 1.0
	 */
	public static Utf8ResourceBundle getUtf8Bundle(String baseName, Locale locale) {
		Contract.checkArgument(!Strings.isNullOrEmpty(baseName), "Base name must not be null or empty");
		Contract.checkArgument(locale != null, "Locale must not be null");

		return (Utf8ResourceBundle) getBundle(baseName, locale, new Control());
	}

	/**
	 * Returns a {@code Utf8ResourceBundle} located using the given base name, locale, and class loader.
	 *
	 * @throws IllegalArgumentException if {@code baseName} is {@code null} or empty
	 * @throws IllegalArgumentException if {@code locale} is {@code null}
	 * @throws IllegalArgumentException if {@code classLoader} is {@code null}
	 *
	 * @see ResourceBundle#getBundle(String, Locale, ClassLoader)
	 *
	 * @since 1.0
	 */
	public static Utf8ResourceBundle getUtf8Bundle(String baseName, Locale locale, ClassLoader classLoader) {
		Contract.checkArgument(!Strings.isNullOrEmpty(baseName), "Base name must not be null or empty");
		Contract.checkArgument(locale != null, "Locale must not be null");
		Contract.checkArgument(classLoader != null, "Class loader must not be null");

		return (Utf8ResourceBundle) getBundle(baseName, locale, new Control());
	}


	HashMap<String, String> toMap() {
		HashMap<String, String> map = new HashMap<>();
		for (Enumeration<String> keys = getKeys(); keys.hasMoreElements(); ) {
			String key = keys.nextElement();
			map.put(key, getString(key));
		}

		return map;
	}

	// ------------------------------------------------------------
	// ------------------------------------------------------------
	// ------------------------------------------------------------

	private static class Control extends ResourceBundle.Control {

		@Override
		public List<String> getFormats(String baseName) {
			return Collections.unmodifiableList(Arrays.asList(SUFFIX));
		}

		@Override
		public ResourceBundle newBundle(String baseName, Locale locale, String format, ClassLoader classLoader, boolean reload) throws IOException {
			String resourceName = toResourceName(toBundleName(baseName, locale), SUFFIX);

			InputStream sourceInputStream = classLoader.getResourceAsStream(resourceName);
			if (sourceInputStream == null) {
				return null;
			}

			try (Reader sourceReader = new InputStreamReader(sourceInputStream, UTF_8)) {
				return new Utf8ResourceBundle(sourceReader);
			}
		}

	}

}
