/*
 * Decompiled with CFR 0.152.
 */
package org.sellcom.core.io;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.UUID;
import org.sellcom.core.Contract;

public class Io {
    private static int bufferSize = 8192;

    private Io() {
    }

    public static void close(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static int copyBytes(InputStream source, OutputStream destination) throws IOException {
        int bytesRead;
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        int bytesCopied = 0;
        byte[] readBuffer = new byte[bufferSize];
        while ((bytesRead = source.read(readBuffer)) != -1) {
            destination.write(readBuffer, 0, bytesRead);
            bytesCopied += bytesRead;
        }
        return bytesCopied;
    }

    public static void copyBytes(InputStream source, OutputStream destination, int byteCount) throws IOException {
        int bytesRead;
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        if (byteCount < 0) {
            Io.copyBytes(source, destination);
            return;
        }
        byte[] readBuffer = new byte[bufferSize];
        for (int bytesCopied = 0; bytesCopied < byteCount; bytesCopied += bytesRead) {
            bytesRead = source.read(readBuffer, 0, Math.min(byteCount - bytesCopied, bufferSize));
            if (bytesRead != -1) {
                destination.write(readBuffer, 0, bytesRead);
                continue;
            }
            throw new EOFException(String.format("Expected %d bytes but got only %d", byteCount, bytesCopied));
        }
    }

    public static int discardBytes(InputStream source) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        return Io.copyBytes(source, new DiscardingOutputStream());
    }

    public static void discardBytes(InputStream source, int byteCount) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Io.copyBytes(source, new DiscardingOutputStream(), byteCount);
    }

    public static int getBufferSize() {
        return bufferSize;
    }

    public static byte readByte(InputStream source) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        return (byte)Io.readUnsignedByte(source);
    }

    public static byte[] readBytes(InputStream source) throws IOException {
        ByteArrayOutputStream destination = new ByteArrayOutputStream();
        Io.copyBytes(source, destination);
        return destination.toByteArray();
    }

    public static byte[] readBytes(InputStream source, int byteCount) throws IOException {
        ByteArrayOutputStream destination = new ByteArrayOutputStream();
        Io.copyBytes(source, destination, byteCount);
        return destination.toByteArray();
    }

    public static char readChar(InputStream source) throws IOException {
        return (char)Io.readShort(source, ByteOrder.BIG_ENDIAN);
    }

    public static char readChar(InputStream source, ByteOrder byteOrder) throws IOException {
        return (char)Io.readShort(source, byteOrder);
    }

    public static double readDouble(InputStream source) throws IOException {
        return Double.longBitsToDouble(Io.readLong(source, ByteOrder.BIG_ENDIAN));
    }

    public static double readDouble(InputStream source, ByteOrder byteOrder) throws IOException {
        return Double.longBitsToDouble(Io.readLong(source, byteOrder));
    }

    public static float readFloat(InputStream source) throws IOException {
        return Float.intBitsToFloat(Io.readInt(source, ByteOrder.BIG_ENDIAN));
    }

    public static float readFloat(InputStream source, ByteOrder byteOrder) throws IOException {
        return Float.intBitsToFloat(Io.readInt(source, byteOrder));
    }

    public static int readInt(InputStream source) throws IOException {
        return Io.readInt(source, ByteOrder.BIG_ENDIAN);
    }

    public static int readInt(InputStream source, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        int value = 0;
        value |= Io.readUnsignedByte(source);
        value <<= 8;
        value |= Io.readUnsignedByte(source);
        value <<= 8;
        value |= Io.readUnsignedByte(source);
        value <<= 8;
        return byteOrder.equals(ByteOrder.LITTLE_ENDIAN) ? Integer.reverseBytes(value) : (value |= Io.readUnsignedByte(source));
    }

    public static long readLong(InputStream source) throws IOException {
        return Io.readLong(source, ByteOrder.BIG_ENDIAN);
    }

    public static long readLong(InputStream source, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        long value = 0L;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        value |= (long)Io.readUnsignedByte(source);
        value <<= 8;
        return byteOrder.equals(ByteOrder.LITTLE_ENDIAN) ? Long.reverseBytes(value) : (value |= (long)Io.readUnsignedByte(source));
    }

    public static short readShort(InputStream source) throws IOException {
        return Io.readShort(source, ByteOrder.BIG_ENDIAN);
    }

    public static short readShort(InputStream source, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        short value = 0;
        value = (short)(value | Io.readUnsignedByte(source));
        value = (short)(value << 8);
        value = (short)(value | Io.readUnsignedByte(source));
        return byteOrder.equals(ByteOrder.LITTLE_ENDIAN) ? Short.reverseBytes(value) : value;
    }

    public static String readString(InputStream source) throws IOException {
        return Io.readString(source, StandardCharsets.UTF_8);
    }

    public static String readString(InputStream source, Charset charset) throws IOException {
        Contract.checkArgument(charset != null, "Charset must not be null", new Object[0]);
        return new String(Io.readBytes(source), charset);
    }

    public static String readString(InputStream source, int byteCount) throws IOException {
        return Io.readString(source, byteCount, StandardCharsets.UTF_8);
    }

    public static String readString(InputStream source, int byteCount, Charset charset) throws IOException {
        Contract.checkArgument(charset != null, "Charset must not be null", new Object[0]);
        return new String(Io.readBytes(source, byteCount), charset);
    }

    public static int readUnsignedByte(InputStream source) throws IOException {
        Contract.checkArgument(source != null, "Source stream must not be null", new Object[0]);
        int value = source.read();
        if (value == -1) {
            throw new EOFException();
        }
        return value;
    }

    public static UUID readUuid(InputStream source) throws IOException {
        return new UUID(Io.readLong(source, ByteOrder.BIG_ENDIAN), Io.readLong(source, ByteOrder.BIG_ENDIAN));
    }

    public static UUID readUuid(InputStream source, ByteOrder byteOrder) throws IOException {
        return new UUID(Io.readLong(source, byteOrder), Io.readLong(source, byteOrder));
    }

    public static void setBufferSize(int bufferSize) {
        Contract.checkArgument(bufferSize > 0, "Buffer size must be positive: {0}", bufferSize);
        Io.bufferSize = bufferSize;
    }

    public static void writeByte(OutputStream destination, byte value) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        destination.write(value);
    }

    public static void writeBytes(OutputStream destination, byte[] bytes) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        Contract.checkArgument(bytes != null, "Bytes to write must not be null", new Object[0]);
        destination.write(bytes);
    }

    public static void writeChar(OutputStream destination, char value) throws IOException {
        Io.writeChar(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeChar(OutputStream destination, char value, ByteOrder byteOrder) throws IOException {
        Io.writeShort(destination, (short)value, byteOrder);
    }

    public static void writeDouble(OutputStream destination, double value) throws IOException {
        Io.writeDouble(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeDouble(OutputStream destination, double value, ByteOrder byteOrder) throws IOException {
        Io.writeLong(destination, Double.doubleToRawLongBits(value), byteOrder);
    }

    public static void writeFloat(OutputStream destination, float value) throws IOException {
        Io.writeFloat(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeFloat(OutputStream destination, float value, ByteOrder byteOrder) throws IOException {
        Io.writeInt(destination, Float.floatToRawIntBits(value), byteOrder);
    }

    public static void writeInt(OutputStream destination, int value) throws IOException {
        Io.writeInt(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeInt(OutputStream destination, int value, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        if (byteOrder.equals(ByteOrder.BIG_ENDIAN)) {
            value = Integer.reverseBytes(value);
        }
        destination.write(value & 0xFF);
        destination.write((value >>>= 8) & 0xFF);
        destination.write((value >>>= 8) & 0xFF);
        destination.write((value >>>= 8) & 0xFF);
    }

    public static void writeLong(OutputStream destination, long value) throws IOException {
        Io.writeLong(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeLong(OutputStream destination, long value, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        if (byteOrder.equals(ByteOrder.BIG_ENDIAN)) {
            value = Long.reverseBytes(value);
        }
        destination.write((int)value & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
        destination.write((int)(value >>>= 8) & 0xFF);
    }

    public static void writeShort(OutputStream destination, short value) throws IOException {
        Io.writeShort(destination, value, ByteOrder.BIG_ENDIAN);
    }

    public static void writeShort(OutputStream destination, short value, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        Contract.checkArgument(byteOrder != null, "Byte order must not be null", new Object[0]);
        if (byteOrder.equals(ByteOrder.BIG_ENDIAN)) {
            value = Short.reverseBytes(value);
        }
        destination.write(value & 0xFF);
        value = (short)(value >>> 8);
        destination.write(value & 0xFF);
    }

    public static void writeString(OutputStream destination, String string) throws IOException {
        Io.writeString(destination, string, StandardCharsets.UTF_8);
    }

    public static void writeString(OutputStream destination, String string, Charset charset) throws IOException {
        Contract.checkArgument(destination != null, "Destination strema must not be null", new Object[0]);
        Contract.checkArgument(string != null, "String to write must not be null", new Object[0]);
        Contract.checkArgument(charset != null, "Charset must not be null", new Object[0]);
        destination.write(string.getBytes(charset));
    }

    public static void writeUnsignedByte(OutputStream destination, int value) throws IOException {
        Contract.checkArgument(destination != null, "Destination stream must not be null", new Object[0]);
        destination.write(value);
    }

    public static void writeUuid(OutputStream destination, UUID uuid) throws IOException {
        Contract.checkArgument(uuid != null, "UUID to write must not be null", new Object[0]);
        Io.writeLong(destination, uuid.getMostSignificantBits(), ByteOrder.BIG_ENDIAN);
        Io.writeLong(destination, uuid.getLeastSignificantBits(), ByteOrder.BIG_ENDIAN);
    }

    public static void writeUuid(OutputStream destination, UUID uuid, ByteOrder byteOrder) throws IOException {
        Contract.checkArgument(uuid != null, "UUID to write must not be null", new Object[0]);
        Io.writeLong(destination, uuid.getMostSignificantBits(), byteOrder);
        Io.writeLong(destination, uuid.getLeastSignificantBits(), byteOrder);
    }

    private static class DiscardingOutputStream
    extends OutputStream {
        private DiscardingOutputStream() {
        }

        @Override
        public void write(int byteValue) throws IOException {
        }
    }
}

