/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.core.util.platform;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.sellcom.core.Contract;
import org.sellcom.core.Strings;

// see http://www.oracle.com/technetwork/java/javase/versioning-naming-139433.html

/**
 * Version of the Java Runtime Environment (JRE).
 *
 * @since 1.0
 */
public class JreVersion implements Comparable<JreVersion> {

	private static final Pattern VERSION_FORMAT_PATTERN = Pattern.compile("^([0-9]+)\\.([0-9]+)\\.([0-9]+)(_([0-9]+))?(-b([0-9]+))?(-[0-9A-Za-z])?$");

	private final int build;

	private final int major;

	private final int minor;

	private final int revision;

	private final int update;


	private JreVersion(int major, int minor, int revision, int update, int build) {
		this.major = major;
		this.minor = minor;
		this.revision = revision;
		this.update = update;
		this.build = build;
	}


	@Override
	public int compareTo(JreVersion other) {
		Contract.checkArgument(other != null, "Compared version must not be null");

		if (major < other.major) {
			return -1;
		}
		if (major > other.major) {
			return +1;
		}

		if (minor < other.minor) {
			return -1;
		}
		if (minor > other.minor) {
			return +1;
		}

		if (revision < other.revision) {
			return -1;
		}
		if (revision > other.revision) {
			return +1;
		}

		if (update < other.update) {
			return -1;
		}
		if (update > other.update) {
			return +1;
		}

		if (build < other.build) {
			return -1;
		}
		if (build > other.build) {
			return +1;
		}

		return 0;
	}

	/**
	 * Returns the version of the current Java Runtime Environment (JRE).
	 *
	 * @since 1.0
	 */
	public static JreVersion current() {
		return parse(System.getProperty("java.runtime.version"));
	}

	@Override
	public boolean equals(Object other) {
		if (other == this) {
			return true;
		}

		if (other instanceof JreVersion) {
			JreVersion otherCast = (JreVersion) other;

			return Objects.equals(major, otherCast.major)
				&& Objects.equals(minor, otherCast.minor)
				&& Objects.equals(revision, otherCast.revision)
				&& Objects.equals(update, otherCast.update)
				&& Objects.equals(build, otherCast.build);
		}

		return false;
	}

	/**
	 * Returns the build number of this version.
	 *
	 * @since 1.0
	 */
	public int getBuild() {
		return build;
	}

	/**
	 * Returns the major version number of this version.
	 *
	 * @since 1.0
	 */
	public int getMajor() {
		return major;
	}

	/**
	 * Returns the minor version number of this version.
	 *
	 * @since 1.0
	 */
	public int getMinor() {
		return minor;
	}

	/**
	 * Returns the revision number of this version.
	 *
	 * @since 1.0
	 */
	public int getRevision() {
		return revision;
	}

	/**
	 * Returns the update number of this version.
	 *
	 * @since 1.0
	 */
	public int getUpdate() {
		return update;
	}

	@Override
	public int hashCode() {
		return Objects.hash(major, minor, revision, update, build);
	}

	/**
	 * Parses the given version.
	 *
	 * @throws IllegalArgumentException if {@code string} is {@code null} or empty
	 * @throws JreVersionParseException if the version cannot be parsed
	 *
	 * since 1.0
	 */
	public static JreVersion parse(String string) {
		Contract.checkArgument(!Strings.isNullOrEmpty(string), "Input string must not be null or empty");

		Matcher matcher = VERSION_FORMAT_PATTERN.matcher(string);
		if (!matcher.matches()) {
			throw new JreVersionParseException(String.format("Unsupported version format: %s", string));
		}

		try {
			String majorString = matcher.group(1);
			int major = Integer.parseInt(majorString);

			String minorString = matcher.group(2);
			int minor = Integer.parseInt(minorString);

			String revisionString = matcher.group(3);
			int revision = Integer.parseInt(revisionString);

			String updateString = matcher.group(5);
			int update = (Strings.isNullOrEmpty(updateString)) ? -1 : Integer.parseInt(updateString);

			String buildString = matcher.group(7);
			int build = (Strings.isNullOrEmpty(buildString)) ? -1 : Integer.parseInt(buildString);

			return new JreVersion(major, minor, revision, update, build);
		} catch (NumberFormatException e) {
			throw new JreVersionParseException(String.format("Unsupported version format: %s", string));
		}
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append(major);
		builder.append('.');
		builder.append(minor);
		builder.append('.');
		builder.append(revision);

		if (update >= 0) {
			builder.append('_');
			builder.append(update);
		}

		if (build >= 0) {
			builder.append("-b");
			builder.append(build);
		}

		return builder.toString();
	}

}
