/*
 * Copyright (c) 2015-2017 Petr Zelenka <petr.zelenka@sellcom.org>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.sellcom.javafx.beans.binding;

import org.sellcom.core.Contract;

import javafx.beans.binding.Bindings;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.value.ObservableBooleanValue;

/**
 * More common bindings.
 *
 * @since 1.0
 */
public class MoreBindings {

	private MoreBindings() {
		// Utility class, not to be instantiated
	}


	/**
	 * Creates a boolean binding that calculates the conditional-AND operation from the given values.
	 *
	 * @throws IllegalArgumentException if {@code firstValue} is {@code null}
	 * @throws IllegalArgumentException if {@code secondValue} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BooleanBinding and(ObservableBooleanValue firstValue, ObservableBooleanValue secondValue, ObservableBooleanValue... otherValues) {
		Contract.checkArgument(firstValue != null, "First value must not be null");
		Contract.checkArgument(secondValue != null, "Second value must not be null");

		BooleanBinding result = Bindings.and(firstValue, secondValue);
		if (otherValues != null) {
			for (ObservableBooleanValue currentValue : otherValues) {
				result = Bindings.and(result, currentValue);
			}
		}

		return result;
	}

	/**
	 * Creates a boolean binding that calculates the conditional-OR operation from the given values.
	 *
	 * @throws IllegalArgumentException if {@code firstValue} is {@code null}
	 * @throws IllegalArgumentException if {@code secondValue} is {@code null}
	 *
	 * @since 1.0
	 */
	public static BooleanBinding or(ObservableBooleanValue firstValue, ObservableBooleanValue secondValue, ObservableBooleanValue... otherValues) {
		Contract.checkArgument(firstValue != null, "First value must not be null");
		Contract.checkArgument(secondValue != null, "Second value must not be null");

		BooleanBinding result = Bindings.or(firstValue, secondValue);
		if (otherValues != null) {
			for (ObservableBooleanValue currentValue : otherValues) {
				result = Bindings.or(result, currentValue);
			}
		}

		return result;
	}

}
